<?php

namespace Mainto\Api\WorkOrder\WorkOrder\Api;

use Mainto\Api\WorkOrder\WorkOrder\Cmd\CreateByAdminCmd;
use Mainto\Api\WorkOrder\WorkOrder\Cmd\CreateCmd;
use Mainto\Api\WorkOrder\WorkOrder\Cmd\FinishCmd;
use Mainto\Api\WorkOrder\WorkOrder\Cmd\ListCmd;
use Mainto\Api\WorkOrder\WorkOrder\DTO\AcceptorCountDTO;
use Mainto\Api\WorkOrder\WorkOrder\DTO\CategoryCountByDateDTO;
use Mainto\Api\WorkOrder\WorkOrder\DTO\CategoryCountDTO;
use Mainto\Api\WorkOrder\WorkOrder\DTO\CodingInfoDTO;
use Mainto\Api\WorkOrder\WorkOrder\DTO\ListDTO;
use Mainto\Api\WorkOrder\WorkOrder\DTO\PingCodeInfoDTO;
use Mainto\Api\WorkOrder\WorkOrder\DTO\WorkOrderDTO;
use Mainto\RpcServer\RpcAnnotations\Alias;
use Mainto\RpcServer\RpcAnnotations\RpcApi;
use Mainto\RpcServer\RpcAnnotations\RpcAuthority;
use Mainto\RpcServer\RpcAnnotations\RpcCron;
use Mainto\RpcServer\RpcAnnotations\RpcMessageHook;
use Mainto\RpcServer\RpcAnnotations\RpcParam;
use Mainto\RpcServer\RpcServer\RpcContext;
use Mainto\RpcServer\Util\Types\Map;
use Throwable;

interface WorkOrderApi {
    /**
     * @RpcApi(method="POST", url="/work_order/create")
     * @RpcAuthority(need="manager.base")
     * @param CreateCmd $cmd
     * @return int
     */
    public function create (CreateCmd $cmd): int;

    /**
     * @RpcApi(method="GET", url="/work_order/count")
     * @RpcAuthority(need="manager.base")
     */
    public function count (): int;

    /**
     * @Alias(name="获取上传签名")
     * @RpcApi(method="GET", url="/work_order/get_upload_token")
     * @RpcAuthority(need="manager.base")
     * @return string
     */
    public function getUploadToken (): string;

    /**
     * @RpcApi(method="GET", url="/work_order/detail")
     * @RpcParam(name="workOrderId", type="uint", comment="工单ID", require=true)
     * @RpcAuthority(need="manager.base")
     * @param int $workOrderId
     * @return WorkOrderDTO
     */
    public function detail (int $workOrderId): WorkOrderDTO;

    /**
     * @RpcApi(method="GET", url="/work_order/reporter_count")
     * @RpcAuthority(need="manager.base")
     */
    public function reporterCount (): int;

    /**
     * @RpcApi(method="GET", url="/work_order/not_finished_count")
     * @RpcAuthority(need="manager.base")
     */
    public function notFinishedCount (): int;

    /**
     * @RpcApi(method="GET", url="/work_order/bug_count")
     * @RpcAuthority(need="manager.base")
     */
    public function bugCount (): int;

    /**
     * @RpcApi(method="GET", url="/work_order/coding_info")
     * @RpcParam(name="workOrderId", type="uint", comment="工单ID", require=true)
     * @RpcAuthority(need="manager.base")
     *
     * @param int $workOrderId
     * @return CodingInfoDTO|null
     */
    public function codingInfo (int $workOrderId): ?CodingInfoDTO;

    /**
     * @RpcApi(method="POST", url="/work_order/list")
     * @RpcAuthority(need="manager.base")
     * @param ListCmd $command
     * @return ListDTO<WorkOrderDTO>
     */
    public function list (ListCmd $command): ListDTO;

    /**
     * @Alias(name="跳转到coding")
     * @RpcApi(method="GET", url="/work_order/location_to_coding")
     * @RpcParam(name="workOrderId", type="uint", comment="工单ID", require=true)
     * @RpcAuthority(need="manager.base")
     * @param RpcContext $context
     * @param int $workOrderId
     * @return array
     */
    public function locationToCoding (RpcContext $context, int $workOrderId): array;

    /**
     * @Alias(name="跳转到钉钉伙伴")
     * @RpcApi(method="GET", url="/work_order/location_to_ding_chat")
     * @RpcParam(name="workOrderId", type="uint", comment="工单ID", require=true)
     * @RpcAuthority(need="manager.base")
     * @param RpcContext $context
     * @param int $workOrderId
     * @return array
     * @throws Throwable
     */
    public function locationToDingChat (RpcContext $context, int $workOrderId): array;

    /**
     * @Alias(name="跳转到钉钉群组")
     * @RpcApi(method="GET", url="/work_order/location_to_ding_group")
     * @RpcParam(name="workOrderId", type="uint", comment="工单ID", require=true)
     * @RpcAuthority(need="manager.base")
     * @param RpcContext $context
     * @param int $workOrderId
     * @return array
     * @throws Throwable
     */
    public function locationToDingGroup (RpcContext $context, int $workOrderId): array;

    /**
     * @Alias(name="工单关联钉钉群组")
     * @RpcApi(method="GET", url="/work_order/relation_ding_group")
     * @RpcParam(name="workOrderId", type="uint", comment="工单ID", require=true)
     * @RpcParam(name="dingGroupId", type="string", comment="钉钉群组id", require=true)
     * @RpcAuthority(need="manager.base")
     * @param RpcContext $context
     * @param int $workOrderId
     * @param string $dingGroupId
     * @return bool
     */
    public function relationDingGroup (RpcContext $context, int $workOrderId, string $dingGroupId): bool;

    /**
     * @Alias(name="监听xiaobai 卡片点击事件")
     * @RpcMessageHook(name="himo_micro_notify.dingtalk.interactive_card.xiaobai")
     */
    public function hookInteractiveCard (RpcContext $context): string;


    /**
     * @Alias(name="监听场景群回调事件")
     * @RpcMessageHook(name="himo-micro-notify.dingtalk.chat_call_back")
     * @RpcParam(name="eventType", type="string", comment="事件类型", require=true)
     * @RpcParam(name="chatId",type="string", comment="钉钉群组chat id", require=true)
     * @return bool
     */
    public function hookDingtalkChatCallBack (RpcContext $context, string $eventType, string $chatId): bool;

    /**
     * @RpcMessageHook(name="work-order:dingtalk_join_group")
     * @RpcParam(name="workOrderId",type="uint", comment="钉钉群组chat id", require=true)
     * @param int $workOrderId
     * @return bool
     */
    public function hookDingtalkJoinGroup (int $workOrderId): bool;


    /**
     * @RpcMessageHook(name="work-order:dingtalk_dissolution_group")
     * @RpcParam(name="workOrderId",type="uint", comment="钉钉群组chat id", require=true)
     * @RpcParam(name="openConversationId",type="string", comment="钉钉群组openConversationId", require=true)
     * @RpcParam(name="chartGroupId",type="string", comment="钉钉群组Id", require=true)
     * @RpcParam(name="operator", type="uint", comment="operator", require=true)
     * @param int $workOrderId
     * @param string $openConversationId
     * @param int $operator
     * @return bool
     */
    public function hookDingtalkDissolutionGroup (int $workOrderId, string $chartGroupId, string $openConversationId, int $operator): bool;

    /**
     * @RpcApi(method="GET", url="/work_order/categories_count")
     * @RpcParam(name="createdAtStart", type="string", comment="创建时间-起", validation="date_format:Y-m-d")
     * @RpcParam(name="createdAtEnd", type="string", comment="创建时间-止", validation="date_format:Y-m-d")
     * @RpcAuthority(need="manager.base")
     * @param RpcContext $context
     * @return Map<CategoryCountDTO>
     */
    public function categoriesCount (RpcContext $context): Map;

    /**
     * @RpcApi(method="GET", url="/work_order/categories_count_by_date")
     * @RpcParam(name="createdAtStart", type="string", comment="创建时间-起", validation="date_format:Y-m-d")
     * @RpcParam(name="createdAtEnd", type="string", comment="创建时间-止", validation="date_format:Y-m-d")
     * @RpcAuthority(need="manager.base")
     * @return CategoryCountByDateDTO[]
     */
    public function categoriesCountByDate (RpcContext $context): array;

    /**
     * @RpcApi(method="GET", url="/work_order/get_allow_relation_coding")
     * @RpcParam(name="workOrderId", type="uint", comment="工单ID", require=true)
     * @RpcParam(name="codingId", type="uint", comment="codingId", require=true)
     * @RpcAuthority(need="manager.base")
     *
     * @param int $workOrderId
     * @param int $codingId
     * @return CodingInfoDTO|null
     */
    public function getAllowRelationCoding (int $workOrderId, int $codingId): ?CodingInfoDTO;

    // ------------------------------------------------------------------------------------------------------------- //

    /**
     * @RpcApi(method="POST", url="/work_order/create_by_admin")
     * @RpcAuthority(need="manage.999.work_order.manage")
     * @param CreateByAdminCmd $command
     * @return int
     */
    public function createByAdmin (CreateByAdminCmd $command): int;

    /**
     * @RpcApi(method="POST", url="/work_order/delete")
     * @RpcParam(name="workOrderId", type="uint", comment="工单ID", require=true)
     * @RpcParam(name="operator", type="uint", comment="operator", require=true, fromSession="manage_staffid")
     * @RpcAuthority(need="manage.999.work_order.manage")
     * @param int $workOrderId
     * @param int $operator
     * @return bool
     */
    public function delete (int $workOrderId, int $operator): bool;

    /**
     * @RpcApi(method="POST", url="/work_order/processing")
     * @RpcParam(name="workOrderId", type="uint", comment="工单ID", require=true)
     * @RpcParam(name="operator", type="uint", comment="operator", require=true, fromSession="manage_staffid")
     * @RpcAuthority(need="manage.999.work_order.manage")
     * @param int $workOrderId
     * @param int $operator
     * @return WorkOrderDTO
     * @throws Throwable
     */
    public function process (int $workOrderId, int $operator): WorkOrderDTO;

    /**
     * @RpcApi(method="POST", url="/work_order/finish")
     * @RpcAuthority(need="manage.999.work_order.manage")
     * @param FinishCmd $command
     * @return WorkOrderDTO
     * @throws Throwable
     */
    public function finish (FinishCmd $command): WorkOrderDTO;

    /**
     * @RpcApi(method="GET", url="/work_order/categories")
     * @RpcAuthority(need="manager.base")
     * @return Map<int, string>
     */
    public function categories (): Map;

    /**
     * @RpcApi(method="POST", url="/work_order/update_category")
     * @RpcParam(name="workOrderId", type="uint", comment="工单ID", require=true)
     * @RpcParam(name="categoryId", type="uint", comment="分类", require=true)
     * @RpcParam(name="operator", type="uint", comment="operator", require=true, fromSession="manage_staffid")
     * @RpcAuthority(need="manage.999.work_order.manage")
     * @param int $workOrderId
     * @param int $categoryId
     * @param int $operator
     * @return WorkOrderDTO
     * @throws Throwable
     */
    public function updateCategory (int $workOrderId, int $categoryId, int $operator): WorkOrderDTO;

    /**
     * @RpcApi(method="POST", url="/work_order/update_current_acceptor")
     * @RpcParam(name="workOrderId", type="uint", comment="工单ID", require=true)
     * @RpcParam(name="currentAcceptor", type="uint", comment="处理人", require=true)
     * @RpcAuthority(need="manage.999.work_order.manage")
     * @param int $workOrderId
     * @param int $currentAcceptor
     * @return WorkOrderDTO
     * @throws Throwable
     */
    public function updateCurrentAcceptor (int $workOrderId, int $currentAcceptor): WorkOrderDTO;

    /**
     * @RpcApi(method="POST", url="/work_order/create_coding")
     * @RpcParam(name="workOrderId", type="uint", comment="工单ID", require=true)
     * @RpcAuthority(need="manage.999.work_order.manage")
     * @param RpcContext $context
     * @param int $workOrderId
     * @return WorkOrderDTO
     */
    public function createCoding (RpcContext $context, int $workOrderId): WorkOrderDTO;

    /**
     * @RpcApi(method="POST", url="/work_order/bind_coding")
     * @RpcParam(name="workOrderId", type="uint", comment="工单ID", require=true)
     * @RpcParam(name="codingId", type="uint", comment="codingId", require=true)
     * @RpcAuthority(need="manage.999.work_order.manage")
     * @param int $workOrderId
     * @param int $codingId
     * @return WorkOrderDTO
     */
    public function bindCoding (int $workOrderId, int $codingId): WorkOrderDTO;

    /**
     * @RpcApi(method="GET", url="/work_order/reset_deleted_coding")
     * @RpcParam(name="createdAtStart", type="string", comment="创建时间-起", validation="date_format:Y-m-d")
     * @RpcParam(name="createdAtEnd", type="string", comment="创建时间-止", validation="date_format:Y-m-d")
     * @RpcAuthority(need="manage.999.work_order.manage")
     * @RpcCron(every="12h")
     * @param RpcContext $context
     * @throws Throwable
     */
    public function resetDeletedCoding (RpcContext $context): void;

    /**
     * @RpcApi(method="POST", url="/work_order/remove_coding_relation")
     * @RpcParam(name="workOrderId", type="uint", comment="工单ID", require=true)
     * @RpcParam(name="operator", type="uint", comment="operator", require=true, fromSession="manage_staffid")
     * @RpcAuthority(need="manage.999.work_order.manage")
     */
    public function removeCodingRelation (int $workOrderId, int $operator): bool;

    /**
     * @RpcApi(method="GET", url="/work_order/acceptor_count")
     * @RpcParam(name="createdAtStart", type="string", comment="创建时间-起", validation="date_format:Y-m-d")
     * @RpcParam(name="createdAtEnd", type="string", comment="创建时间-止", validation="date_format:Y-m-d")
     * @RpcAuthority(need="manager.base")
     * @param RpcContext $context
     * @return Map<AcceptorCountDTO>
     */
    public function acceptorCount (RpcContext $context): Map;


    /**
     * @RpcApi(method="POST", url="/work_order/create_ping_code")
     * @RpcParam(name="workOrderId", type="uint", comment="工单ID", require=true)
     * @RpcAuthority(need="manage.999.work_order.manage")
     * @param RpcContext $context
     * @param int $workOrderId
     * @return WorkOrderDTO
     */
    public function createPingCode (RpcContext $context, int $workOrderId): WorkOrderDTO;

    /**
     * @RpcApi(method="POST", url="/work_order/bind_ping_code")
     * @RpcParam(name="workOrderId", type="uint", comment="工单ID", require=true)
     * @RpcParam(name="pingCodeId", type="string", comment="pingCodeId", require=true)
     * @RpcAuthority(need="manage.999.work_order.manage")
     * @param int $workOrderId
     * @param string $pingCodeId
     * @return WorkOrderDTO
     */
    public function bindPingCode (int $workOrderId, string $pingCodeId): WorkOrderDTO;


    /**
     * @RpcApi(method="GET", url="/work_order/reset_deleted_ping_code")
     * @RpcParam(name="createdAtStart", type="string", comment="创建时间-起", validation="date_format:Y-m-d")
     * @RpcParam(name="createdAtEnd", type="string", comment="创建时间-止", validation="date_format:Y-m-d")
     * @RpcAuthority(need="manage.999.work_order.manage")
     * @RpcCron(every="12h")
     * @param RpcContext $context
     * @throws Throwable
     */
    public function resetDeletedPingCode (RpcContext $context): void;

    /**
     * @RpcApi(method="POST", url="/work_order/remove_ping_code_relation")
     * @RpcParam(name="workOrderId", type="uint", comment="工单ID", require=true)
     * @RpcParam(name="operator", type="uint", comment="operator", require=true, fromSession="manage_staffid")
     * @RpcAuthority(need="manage.999.work_order.manage")
     */
    public function removePingCodeRelation (int $workOrderId, int $operator): bool;

    /**
     * @RpcApi(method="GET", url="/work_order/get_allow_relation_ping_code")
     * @RpcParam(name="pingCodeId", type="string", comment="pingCodeId", require=true)
     * @RpcAuthority(need="manager.base")
     *
     * @param string $pingCodeId
     * @return CodingInfoDTO|null
     */
    public function getAllowRelationPingCode (string $pingCodeId): ?PingCodeInfoDTO;


    /**
     * @Alias(name="跳转到pingCode")
     * @RpcApi(method="GET", url="/work_order/location_to_ping_code")
     * @RpcParam(name="workOrderId", type="uint", comment="工单ID", require=true)
     * @RpcAuthority(need="manager.base")
     * @param RpcContext $context
     * @param int $workOrderId
     * @return array
     */
    public function locationToPingCode (RpcContext $context, int $workOrderId): array;

    /**
     * @RpcApi(method="GET", url="/work_order/ping_code_info")
     * @RpcParam(name="workOrderId", type="uint", comment="工单ID", require=true)
     * @RpcAuthority(need="manager.base")
     *
     * @param int $workOrderId
     * @return CodingInfoDTO|null
     */
    public function pingCodeInfo (int $workOrderId): ?PingCodeInfoDTO;
}
