<?php

namespace Mainto\Websocket\Auth;

use Carbon\Carbon;
use Mainto\Websocket\CacheKey;
use Illuminate\Support\Facades\Redis;
use Mainto\Websocket\Contracts\Userable;
use Mainto\Websocket\Contracts\AuthGuard;
use Mainto\RpcServer\RpcServer\RpcContext;
use Mainto\Websocket\Events\ClearLastLoginInfo;
use Mainto\Websocket\Auth\Traits\AuthGuardHelper;

class RedisAuthProvider implements AuthGuard
{
    use AuthGuardHelper;

    /**
     * @var \Illuminate\Redis\Connections\Connection
     */
    protected $redis;

    /**
     * @var string
     */
    protected $keyUuidCacheKey;

    /**
     * @var string
     */
    protected $uuidUserCacheKey;

    /**
     * @var Manager
     */
    protected $auth;

    /**
     * @var RpcContext
     */
    protected $context;

    /**
     * WsUserProvider constructor.
     * @param Manager $auth
     */
    public function __construct(Manager $auth)
    {
        $this->auth = $auth;
        $this->context = $auth->context;
        $this->redis = Redis::connection(config('ws.connection', 'default'));
        $this->uuidUserCacheKey = CacheKey::uuidUserMap($this->room());
        $this->keyUuidCacheKey = CacheKey::keyUuidMap($this->room());
    }

    /**
     * @param Userable $user
     *
     * @author duc <1025434218@qq.com>
     */
    public function login(Userable $user)
    {
        $key = $this->getKey($user);

        if (($oldUuid = $this->redis->hget($this->keyUuidCacheKey, $key)) && $oldUuid != $user->getUuid()) {
            $oldUserInfo = $this->getUserByUuid($oldUuid);
            $this->redis->hdel($this->uuidUserCacheKey, [$oldUuid]);
            $this->redis->hdel($this->keyUuidCacheKey, [$key]);
            event(new ClearLastLoginInfo($oldUserInfo, $this->room()));
        }

        if (!$this->userExists($user) && $user->getUuid() && $this->getKey($user)) {
            $this->cacheLoginUserUuid($user);
            $this->cacheUuidUserMap($user);
        }

        $this->setWebsocketUser($user);
    }

    /**
     * @param Userable|null $user
     *
     * @author duc <1025434218@qq.com>
     */
    public function logout(Userable $user = null)
    {
        $user = $this->getWebsocketUser();

        if ($user) {
            $this->redis->hdel($this->keyUuidCacheKey, [$this->getKey($user)]);
            $this->redis->hdel($this->uuidUserCacheKey, [$user->getUuid()]);
        }
        $this->setWebsocketUser(null);
    }

    /**
     * @param string $uuid
     * @return User|null
     *
     * @author duc <1025434218@qq.com>
     */
    public function getUserByUuid(string $uuid)
    {
        return @unserialize($this->redis->hget($this->uuidUserCacheKey, $uuid)) ?: null;
    }

    /**
     * @param $key
     * @return string
     *
     * @author duc <1025434218@qq.com>
     */
    public function getUuidByKey($key)
    {
        return $this->redis->hget($this->keyUuidCacheKey, $this->getKey($key));
    }

    /**
     * @param Userable $user
     *
     * @author duc <1025434218@qq.com>
     */
    protected function cacheLoginUserUuid(Userable $user): void
    {
        $this->setExpireIfNotExist($this->keyUuidCacheKey, function ($redis) use ($user) {
            $redis->hset($this->keyUuidCacheKey, $this->getKey($user), $user->getUuid());
        });
    }

    /**
     * @param string $uuid
     *
     * @author duc <1025434218@qq.com>
     */
    public function dissociate(string $uuid)
    {
        $this->redis->hdel($this->uuidUserCacheKey, [$uuid]);
    }

    /**
     * @param Userable $user
     *
     * @author duc <1025434218@qq.com>
     */
    protected function cacheUuidUserMap(Userable $user): void
    {
        $this->setExpireIfNotExist($this->uuidUserCacheKey, function ($redis) use ($user) {
            $redis->hset($this->uuidUserCacheKey, $user->getUuid(), serialize($user));
        });
    }

    /**
     * @param string $key
     * @param callable $callback
     * @param int|null $seconds
     * @return mixed
     *
     * @author duc <1025434218@qq.com>
     */
    public function setExpireIfNotExist(string $key, callable $callback, int $seconds = null)
    {
        if (! $this->redis->exists($key)) {
            $res = $callback($this->redis);
            $this->redis->expire($key, $seconds ?? $this->getCacheSeconds());
        } else {
            $res = $callback($this->redis);
        }

        return $res;
    }

    /**
     * @return array
     *
     * @author duc <1025434218@qq.com>
     */
    public function getAllKeyUuid()
    {
        return $this->redis->hgetall($this->keyUuidCacheKey);
    }

    /**
     * @return array
     *
     * @author duc <1025434218@qq.com>
     */
    public function getAllUuidUsers()
    {
        return array_map('unserialize', $this->redis->hgetall($this->uuidUserCacheKey));
    }

    /**
     * @return int
     *
     * @author duc <1025434218@qq.com>
     */
    private function getCacheSeconds()
    {
        $expire = config('ws.auth.expire', 'tomorrow');

        if (is_numeric($expire)) {
            return $expire;
        }

        return now()->diffInRealSeconds(Carbon::parse($expire));
    }

    /**
     * @return \Illuminate\Redis\Connections\Connection
     *
     * @author duc <1025434218@qq.com>
     */
    public function getRedis()
    {
        return $this->redis;
    }

    public function loginExpired(Userable $user)
    {
        if ($u = $this->getUserByUuid($user->getUuid())) {
            return false;
        }

        $this->redis->hdel($this->keyUuidCacheKey, [$this->getKey($user)]);

        return true;
    }

    private function userExists (Userable $user) {
        return $this->redis->hexists($this->keyUuidCacheKey, $this->getKey($user))
            && $this->redis->hexists($this->uuidUserCacheKey, $user->getUuid());
    }
}
