<?php

namespace Mainto\Websocket\Auth;

use Carbon\Carbon;
use Mainto\Websocket\Auth\Models\User;
use Mainto\Websocket\Contracts\Userable;
use Mainto\Websocket\Contracts\AuthGuard;
use Mainto\RpcServer\RpcServer\RpcContext;
use Mainto\Websocket\Events\ClearLastLoginInfo;
use Mainto\Websocket\Auth\Traits\AuthGuardHelper;

class MysqlAuthProvider implements AuthGuard
{
    use AuthGuardHelper;

    /**
     * @var Manager
     */
    protected $auth;

    /**
     * @var RpcContext
     */
    protected $context;

    /**
     * WsUserProvider constructor.
     * @param Manager $auth
     */
    public function __construct(Manager $auth)
    {
        $this->auth = $auth;
        $this->context = $auth->context;
    }

    /**
     * @inheritDoc
     */
    public function login(Userable $user)
    {
        $key = $this->getKey($user);
        if ($oldUser = User::query()
            ->latest('id')
            ->where('room', $this->room())
            ->where('key', $key)
            ->where('uuid', '!=', $user->getUuid())
            ->first()
        ) {
            User::query()->where('room', $this->room())->where('key', $key)->delete();
            event(new ClearLastLoginInfo($oldUser->payload, $this->room()));
        }

        if (
            !$this->userExists($user)
            && $user->getUuid()
            && $this->getKey($user)
        ) {
            User::query()->firstOrCreate([
                'room'           => $this->room(),
                'uuid'           => $user->getUuid(),
                'key'            => $this->getKey($user),
                'expired_at'     => $this->getExpiredAt(),
            ], [
                'payload'        => $user,
            ]);
        }

        $this->setWebsocketUser($user);
    }

    /**
     * @inheritDoc
     */
    public function logout(Userable $user = null)
    {
        $user = $this->getWebsocketUser();

        if ($user) {
            User::query()->where('room', $this->room())->where('uuid', $user->getUuid())->where('key', $this->getKey($user))->delete();
        }

        $this->setWebsocketUser(null);
    }

    /**
     * @inheritDoc
     */
    public function getUserByUuid(string $uuid)
    {
        return User::query()->where('room', $this->room())->where('uuid', $uuid)->value('payload');
    }

    /**
     * @inheritDoc
     */
    public function getUuidByKey($key)
    {
        return optional(User::query()->where('room', $this->room())->where('key', $this->getKey($key))->first())->uuid;
    }

    /**
     * @inheritDoc
     */
    public function dissociate(string $uuid)
    {
        User::query()->where('room', $this->room())->where('uuid', $uuid)->delete();
    }

    /**
     * @inheritDoc
     */
    public function getAllKeyUuid()
    {
        return User::query()->where('room', $this->room())->get()->pluck('uuid', 'key')->toArray();
    }

    /**
     * @inheritDoc
     */
    public function getAllUuidUsers()
    {
        return User::query()->where('room', $this->room())->pluck('payload', 'uuid')->toArray();
    }

    public function loginExpired(Userable $user)
    {
        return optional(User::query()->where('room', $this->room())->where('uuid', $user->getUuid())->first())->loginExpired() ?? true;
    }

    private function getExpiredAt()
    {
        $expire = config('ws.auth.expire', null);
        if (is_numeric($expire) && $expire >= 0) {
            return now()->addSecond($expire);
        }

        return Carbon::parse($expire);
    }

    /**
     * @param Userable $user
     * @return bool
     *
     * @author 神符 <1025434218@qq.com>
     */
    protected function userExists (Userable $user): bool {
        return User::query()
            ->where([
                ['room', $this->room()],
                ['key', $this->getKey($user)],
                ['uuid', $user->getUuid()],
            ])
            ->exists();
    }
}
