<?php

namespace Mainto\Websocket;

use Mainto\Websocket\Contracts\Queueable;
use Mainto\RpcServer\RpcServer\RpcContext;
use Mainto\RpcServer\RpcUtil\RpcMessageQueue;
use Mainto\RpcServer\Exceptions\RpcRuntimeException;
use Mainto\Websocket\Contracts\Response as ResponseImp;

/**
 * Class Response
 * @package Mainto\Websocket
 */
class Response implements ResponseImp, Queueable
{
    const WS_NOTIFY = 'notify';

    protected $once = false;

    public function atOnce(): ResponseImp
    {
        $this->once = true;

        return $this;
    }

    /**
     * @param RpcContext $context
     * @param $data
     * @param string $type
     *
     * @param string $room
     * @author duc <1025434218@qq.com>
     */
    public function successBroadcast(RpcContext $context, $data, string $type = '', string $room = null)
    {
        $type = $this->parseType($context, $type);

        $this->shouldNotAddReturn($context, $type);
        $msg = [
            'success' => true,
            'data'    => $data,
            'type'    => $type,
        ];

        $this->sendIfVerified($context, '[all]', $msg, $room);
    }

    /**
     * @param RpcContext $context
     * @param $data
     * @param string $type
     *
     * @param string $room
     * @author duc <1025434218@qq.com>
     */
    public function successToSelf(RpcContext $context, $data, string $type = '', string $room = null)
    {
        $type = $this->parseType($context, $type);

        $this->shouldNotAddReturn($context, $type);

        $msg = [
            'success' => true,
            'data'    => $data,
            'type'    => $type,
        ];

        $this->sendIfVerified($context, $context->getUserId(), $msg, $room);
    }

    /**
     * @param RpcContext $context
     * @param string|null $uuid
     * @param $data
     * @param string $type
     *
     * @param string $room
     * @author duc <1025434218@qq.com>
     */
    public function successToUser(RpcContext $context, string $uuid, $data, string $type = '', string $room = null)
    {
        $type = $this->parseType($context, $type);

        $this->shouldNotAddReturn($context, $type);

        $msg = [
            'success' => true,
            'data'    => $data,
            'type'    => $type,
        ];

        $this->sendIfVerified($context, $uuid, $msg, $room);
    }

    /**
     * @param RpcContext $context
     * @param \Throwable $exception
     * @param array $appends
     * @param string $type
     *
     * @param string $room
     * @author duc <1025434218@qq.com>
     */
    public function errorBroadcast(RpcContext $context, \Throwable $exception, array $appends = [], string $type = '', string $room = null)
    {
        $type = $this->parseType($context, $type);

        $this->shouldNotAddReturn($context, $type);
        $errorCode = $exception->getCode();
        $errorMsg = $exception->getMessage();

        $msg = [
            'success' => false,
            'code'    => $errorCode,
            'data'    => $appends,
            'msg'     => $errorMsg,
            'type'    => $type,
        ];

        $this->sendIfVerified($context, '[all]', $msg, $room);
    }

    /**
     * @param RpcContext $context
     * @param \Throwable $exception
     * @param array $appends
     * @param string $type
     *
     * @param string $room
     * @author duc <1025434218@qq.com>
     */
    public function errorToSelf(RpcContext $context, \Throwable $exception, array $appends = [], string $type = '', string $room = null)
    {
        $type = $this->parseType($context, $type);

        $this->shouldNotAddReturn($context, $type);
        $errorCode = $exception->getCode();
        $errorMsg = $exception->getMessage();

        $msg = [
            'success' => false,
            'code'    => $errorCode,
            'data'    => $appends,
            'msg'     => $errorMsg,
            'type'    => $type,
        ];

        $this->sendIfVerified($context, $context->getUserId(), $msg, $room);
    }

    /**
     * @param RpcContext $context
     * @param string|null $uuid
     * @param \Throwable $exception
     * @param array $appends
     * @param string $type
     *
     * @param string $room
     * @author duc <1025434218@qq.com>
     */
    public function errorToUser(RpcContext $context, string $uuid, \Throwable $exception, array $appends = [], string $type = '', string $room = null)
    {
        $type = $this->parseType($context, $type);

        $this->shouldNotAddReturn($context, $type);

        $errorCode = $exception->getCode();
        $errorMsg = $exception->getMessage();

        $msg = [
            'success' => false,
            'code'    => $errorCode,
            'data'    => $appends,
            'msg'     => $errorMsg,
            'type'    => $type,
        ];

        $this->sendIfVerified($context, $uuid, $msg, $room);
    }

    /**
     * @param RpcContext $context
     * @param string $notifyType
     * @param array $payload
     * @param string $room
     *
     * @author duc <1025434218@qq.com>
     */
    public function notifyToSelf(RpcContext $context, string $notifyType, array $payload = [], string $room = null)
    {
        $msg = [
            'success' => true,
            'data'    => [
                'notify_type' => $notifyType,
                'payload'     => $payload,
            ],
            'type'    => self::WS_NOTIFY,
        ];

        $this->sendIfVerified($context, $context->getUserId(), $msg, $room);
    }

    /**
     * @param RpcContext $context
     * @param string $notifyType
     * @param string|null $uuid
     * @param array $payload
     * @param string $room
     *
     * @author duc <1025434218@qq.com>
     */
    public function notifyUser(RpcContext $context, string $notifyType, string $uuid = null, array $payload = [], string $room = null)
    {
        $msg = [
            'success' => true,
            'data'    => [
                'notify_type' => $notifyType,
                'payload'     => $payload,
            ],
            'type'    => self::WS_NOTIFY,
        ];

        $uuid = $uuid ?? $context->getUserId();

        $this->sendIfVerified($context, $uuid, $msg, $room);
    }

    /**
     * @param RpcContext $context
     * @param string $notifyType
     * @param array $payload
     * @param string $room
     *
     * @author duc <1025434218@qq.com>
     */
    public function notifyAll(RpcContext $context, string $notifyType, array $payload = [], string $room = null)
    {
        $msg = [
            'success' => true,
            'data'    => [
                'notify_type' => $notifyType,
                'payload'     => $payload,
            ],
            'type'    => self::WS_NOTIFY,
        ];

        $this->sendIfVerified($context, '[all]', $msg, $room);
    }

    /**
     * @param string|null $uuid
     * @param string $room
     * @param array $msg
     *
     * @author duc <1025434218@qq.com>
     */
    public function sendToUser(string $uuid, string $room, array $msg)
    {
        $this->atOnce()->sendIfVerified(null, $uuid, $msg, $room);
    }

    public function push(RpcContext $context, string $clientId, string $roomName, array $msg)
    {
        $contextKey = config('ws.ws_queued_data_key', 'ws_queued_data');
        $oldData = $context->{$contextKey} ?? [];
        $key = $roomName . '|' . $clientId;
        $data = [
            $key => [$msg],
        ];

        $context->{$contextKey} = array_merge_recursive($oldData, $data);
    }

    public function fire(RpcContext $context)
    {
        $key = config('ws.ws_queued_data_key', 'ws_queued_data');
        $items = $context->{$key};
        foreach ($items as $key => $messages) {
            [$room, $clientId] = explode('|', $key);

            $this->send($clientId, $room, $messages, false);
        }
        unset($context->inWebsocketRoom, $context->{$key});
    }

    /**
     * @param RpcContext $context
     * @param string|null $clientId
     * @param $msg
     * @param string|null $room
     *
     * @author duc <1025434218@qq.com>
     */
    protected function sendIfVerified(?RpcContext $context, ?string $clientId, array $msg, ?string $room)
    {
        $roomName = $room ?: optional($context)->getRoomName();
        if ($roomName && $clientId) {
            if ($this->shouldPushQueue($context)) {
                $this->push($context, $clientId, $roomName, $msg);

                return;
            }

            $this->send($clientId, $roomName, $msg, true);
            $this->once && $this->once = false;
        }
    }

    private function send(string $uuid, string $room, array $msg, $singleMode = true)
    {
        app(RpcMessageQueue::class)->addMessageToWebSocket(
            $room,
            $uuid,
            json_encode(
                $singleMode ? [$msg] : $msg,
                JSON_UNESCAPED_UNICODE
            )
        );
    }

    public function shouldPushQueue(RpcContext $context)
    {
        return ! ! config('ws.queued', false) && ! $this->once && ($context->inWebsocketRoom ?? false);
    }

    /**
     * @param RpcContext $context
     * @param string $type
     *
     * @author duc <1025434218@qq.com>
     */
    public function shouldNotAddReturn(RpcContext $context, string $type)
    {
        $type = $this->parseType($context, $type);
        $context->shouldAddReturn[$type] = false;
    }

    /**
     * @param RpcContext $context
     *
     * @author duc <1025434218@qq.com>
     */
    public function resetShouldAddReturn(RpcContext $context)
    {
        $context->shouldAddReturn = [];
    }

    /**
     * @param RpcContext $context
     * @param string $type
     * @return bool
     *
     * @author duc <1025434218@qq.com>
     */
    public function shouldAddReturn(RpcContext $context, string $type)
    {
        $type = $this->parseType($context, $type);

        return $context->shouldAddReturn[$type] ?? true;
    }

    /**
     * @param RpcContext $context
     * @param string $type
     * @return string
     *
     * @author duc <1025434218@qq.com>
     */
    public function parseType(RpcContext $context, string $type = ''): string
    {
        $type = $type ?: ($context->websocketActionType ?? '');

        if ($type) {
            return $type;
        }

        throw new RpcRuntimeException('unknown websocket type!');
    }
}
