<?php

namespace Mainto\Websocket\Traits;

use ReflectionMethod;
use ReflectionParameter;
use Illuminate\Support\Arr;
use ReflectionFunctionAbstract;
use Mainto\RpcServer\RpcServer\RpcContext;
use Illuminate\Contracts\Container\BindingResolutionException;

/**
 * Trait RouteDependencyResolverTrait
 * @package Mainto\Websocket\Traits
 */
trait RouteDependencyResolverTrait
{
    /**
     * Resolve the object method's type-hinted dependencies.
     *
     * @param $context
     * @param array $parameters
     * @param object $instance
     * @param string $method
     * @return array
     * @throws \ReflectionException
     * @throws BindingResolutionException
     */
    protected function resolveClassMethodDependencies($context, array $parameters, $instance, $method)
    {
        if (! method_exists($instance, $method)) {
            return $parameters;
        }

        return $this->resolveMethodDependencies(
            $context,
            $parameters,
            new ReflectionMethod($instance, $method)
        );
    }

    /**
     * Resolve the given method's type-hinted dependencies.
     *
     * @param $context
     * @param array $parameters
     * @param \ReflectionFunctionAbstract $reflector
     * @return array
     * @throws \ReflectionException
     * @throws BindingResolutionException
     */
    public function resolveMethodDependencies($context, array $parameters, ReflectionFunctionAbstract $reflector)
    {
        $result = [];

        foreach ($reflector->getParameters() as $key => $parameter) {
            $result[] = $this->transformDependency(
                $context,
                $parameter,
                $parameters
            );
        }

        return $result;
    }

    /**
     * @param ReflectionParameter $parameter
     * @param array $params
     * @return bool
     *
     * @author 神符 <1025434218@qq.com>
     */
    public function hasParameterOverride(ReflectionParameter $parameter, array $params)
    {
        return isset($params[$parameter->name]);
    }

    /**
     * @param ReflectionParameter $parameter
     * @param array $params
     * @return mixed
     *
     * @author 神符 <1025434218@qq.com>
     */
    public function getParameterOverride(ReflectionParameter $parameter, array $params)
    {
        return $params[$parameter->name];
    }

    /**
     * Attempt to transform the given parameter into a class instance.
     *
     * @param $context
     * @param \ReflectionParameter $parameter
     * @param array $parameters
     * @return mixed
     * @throws BindingResolutionException
     * @throws \ReflectionException
     */
    protected function transformDependency($context, ReflectionParameter $parameter, $parameters)
    {
        if ($this->hasParameterOverride($parameter, $parameters)) {
            return $this->getParameterOverride($parameter, $parameters);
        }

        $class = $parameter->getClass();

        return is_null($class)
            ? $this->resolvePrimitive($parameter)
            : $this->resolveClass($context, $parameter, $parameters);
    }

    /**
     * @param ReflectionParameter $parameter
     * @return mixed
     * @throws BindingResolutionException
     * @throws \ReflectionException
     *
     * @author 神符 <1025434218@qq.com>
     */
    public function resolvePrimitive(ReflectionParameter $parameter)
    {
        if ($parameter->isDefaultValueAvailable()) {
            return $parameter->getDefaultValue();
        }

        $this->unresolvablePrimitive($parameter);
    }

    /**
     * @param ReflectionParameter $parameter
     * @throws BindingResolutionException
     *
     * @author 神符 <1025434218@qq.com>
     */
    protected function unresolvablePrimitive(ReflectionParameter $parameter)
    {
        $message = "Unresolvable dependency resolving [$parameter] in class {$parameter->getDeclaringClass()->getName()}";

        throw new BindingResolutionException($message);
    }

    /**
     * Determine if an object of the given class is in a list of parameters.
     *
     * @param string $class
     * @param array $parameters
     * @return bool
     */
    protected function alreadyInParameters($class, array $parameters)
    {
        return ! is_null(Arr::first($parameters, function ($value) use ($class) {
            return $value instanceof $class;
        }));
    }

    /**
     * @param $context
     * @param ReflectionParameter $parameter
     * @param $parameters
     * @return mixed
     *
     * @author 神符 <1025434218@qq.com>
     */
    protected function resolveClass($context, ReflectionParameter $parameter, $parameters)
    {
        $contextInjection = config('ws.context_injection', true);

        $class = $parameter->getClass();

        if ($contextInjection && $class->name == RpcContext::class) {
            return $context;
        }

        // If the parameter has a type-hinted class, we will check to see if it is already in
        // the list of parameters. If it is we will just skip it as it is probably a model
        // binding and we do not want to mess with those; otherwise, we resolve it here.
        if (! $this->alreadyInParameters($class->name, $parameters)) {
            if ($contextInjection) {
                $injections = collect($class->getConstructor()->getParameters())->filter(function (ReflectionParameter $p) use ($parameters) {
                    $sub = $p->getClass();
                    if ($sub && $sub->name == RpcContext::class) {
                        if (isset($parameters[$p->name])) {
                            return ! $parameters[$p->name] instanceof RpcContext;
                        }

                        return true;
                    }

                    return false;
                })->mapWithKeys(function ($name) use ($context) {
                    return [$name->name => $context];
                })->toArray();

                $classObj = app()->make($class->name, $parameters + $injections);
            } else {
                $classObj = app()->make($class->name, $parameters);
            }

            $this->fireAfterResolvingCallbacks($class->name, $classObj);

            return $classObj;
        }
    }

    public function fireAfterResolvingCallbacks($abstract, $object)
    {
    }
}
