<?php

namespace Mainto\Websocket\Auth\Traits;

use Mainto\Websocket\CacheKey;
use Mainto\Websocket\Contracts\Userable;
use Illuminate\Validation\UnauthorizedException;
use Mainto\RpcServer\Exceptions\RpcRuntimeException;

trait AuthGuardHelper
{
    public function room()
    {
        if ($room = $this->context->getRoomName()) {
            return $room;
        }

        throw new RpcRuntimeException();
    }

    /**
     * @return Userable|null
     *
     * @author duc <1025434218@qq.com>
     */
    public function getWebsocketUser()
    {
        return $this->context->websocketUser ?? null;
    }

    /**
     * @param Userable|null $user
     *
     * @author duc <1025434218@qq.com>
     */
    public function setWebsocketUser(?Userable $user): void
    {
        $this->context->websocketUser = $user;
    }

    /**
     * Set the current user.
     *
     * @param string $uuid
     * @return Userable|null
     */
    public function setUserByUuid(string $uuid)
    {
        $user = $this->getUserByUuid($uuid);

        if (! $user) {
            return null;
        }

        $this->login($user);

        return $user;
    }

    /**
     * Get the ID for the currently authenticated user.
     *
     * @return string|null
     */
    public function uuid()
    {
        return $this->user()
            ? $this->user()->getUuid()
            : $this->context->getUserId();
    }

    /**
     * @return bool
     *
     * @author duc <1025434218@qq.com>
     */
    public function hasUser()
    {
        return ! ! $this->user();
    }

    /**
     * Determine if the current user is authenticated.
     *
     * @return bool
     */
    public function check()
    {
        return ! $this->guest();
    }

    /**
     * @param Userable $user
     * @return bool
     *
     * @author duc <1025434218@qq.com>
     */
    public function isLogin(Userable $user)
    {
        return in_array($user->getUuid(), array_keys($this->getAllUuidUsers()));
    }

    /**
     * @return Userable|null
     *
     * @author duc <1025434218@qq.com>
     */
    public function user()
    {
        $user = $this->getWebsocketUser();

        if ($user) {
            if ($this->isLogin($user) && ! $this->loginExpired($user)) {
                return $user;
            }

            throw new UnauthorizedException();
        }

        if (! $user) {
            $user = $this->setUserByUuid($this->context->getUserId());
        }

        return $user;
    }

    /**
     * Determine if the current user is a guest.
     *
     * @return bool
     */
    public function guest()
    {
        $uuid = $this->context->getUserId();

        $user = $this->getUserByUuid($uuid);

        if ($user && $user->getUuid() == $this->getUuidByKey($user->getKey())) {
            $this->login($user);

            return false;
        }

        $this->dissociate($uuid);

        return true;
    }

    /**
     * @param $key
     * @return string
     *
     * @author duc <1025434218@qq.com>
     */
    protected function getKey($key)
    {
        if ($key instanceof Userable) {
            $key = $key->getKey();
        }

        return CacheKey::userKey($key);
    }
}
