<?php

namespace Mainto\Websocket;

use Mainto\Websocket\Contracts\Queueable;
use Mainto\RpcServer\RpcServer\RpcContext;
use Mainto\RpcServer\RpcUtil\RpcMessageQueue;
use Mainto\RpcServer\Exceptions\RpcRuntimeException;
use Mainto\Websocket\Contracts\Response as ResponseImp;

/**
 * Class Response
 * @package Mainto\Websocket
 */
class Response implements ResponseImp, Queueable
{
    const WS_NOTIFY = 'notify';

    /**
     * @param RpcContext $context
     * @param $data
     * @param string $type
     *
     * @param string $room
     * @author duc <1025434218@qq.com>
     */
    public function successBroadcast(RpcContext $context, $data, string $type = '', string $room = null)
    {
        $type = $this->parseType($context, $type);

        $this->shouldNotAddReturn($context, $type);
        $msg = json_encode([
            'success' => true,
            'data'    => $data,
            'type'    => $type,
        ], JSON_UNESCAPED_UNICODE);

        $this->send($context, '[all]', $msg, $room);
    }

    /**
     * @param RpcContext $context
     * @param $data
     * @param string $type
     *
     * @param string $room
     * @author duc <1025434218@qq.com>
     */
    public function successToSelf(RpcContext $context, $data, string $type = '', string $room = null)
    {
        $type = $this->parseType($context, $type);

        $this->shouldNotAddReturn($context, $type);

        $msg = json_encode([
            'success' => true,
            'data'    => $data,
            'type'    => $type,
        ], JSON_UNESCAPED_UNICODE);

        $this->send($context, $context->getUserId(), $msg, $room);
    }

    /**
     * @param RpcContext $context
     * @param string|null $uuid
     * @param $data
     * @param string $type
     *
     * @param string $room
     * @author duc <1025434218@qq.com>
     */
    public function successToUser(RpcContext $context, ?string $uuid, $data, string $type = '', string $room = null)
    {
        $type = $this->parseType($context, $type);

        $this->shouldNotAddReturn($context, $type);

        $msg = json_encode([
            'success' => true,
            'data'    => $data,
            'type'    => $type,
        ], JSON_UNESCAPED_UNICODE);

        $this->send($context, $uuid, $msg, $room);
    }

    /**
     * @param RpcContext $context
     * @param \Throwable $exception
     * @param array $appends
     * @param string $type
     *
     * @param string $room
     * @author duc <1025434218@qq.com>
     */
    public function errorBroadcast(RpcContext $context, \Throwable $exception, array $appends = [], string $type = '', string $room = null)
    {
        $type = $this->parseType($context, $type);

        $this->shouldNotAddReturn($context, $type);
        $errorCode = $exception->getCode();
        $errorMsg = $exception->getMessage();

        $msg = json_encode([
            'success' => false,
            'code'    => $errorCode,
            'data'    => $appends,
            'msg'     => $errorMsg,
            'type'    => $type,
        ], JSON_UNESCAPED_UNICODE);

        $this->send($context, '[all]', $msg, $room);
    }

    /**
     * @param RpcContext $context
     * @param \Throwable $exception
     * @param array $appends
     * @param string $type
     *
     * @param string $room
     * @author duc <1025434218@qq.com>
     */
    public function errorToSelf(RpcContext $context, \Throwable $exception, array $appends = [], string $type = '', string $room = null)
    {
        $type = $this->parseType($context, $type);

        $this->shouldNotAddReturn($context, $type);
        $errorCode = $exception->getCode();
        $errorMsg = $exception->getMessage();

        $msg = json_encode([
            'success' => false,
            'code'    => $errorCode,
            'data'    => $appends,
            'msg'     => $errorMsg,
            'type'    => $type,
        ], JSON_UNESCAPED_UNICODE);

        $this->send($context, $context->getUserId(), $msg, $room);
    }

    /**
     * @param RpcContext $context
     * @param string|null $uuid
     * @param \Throwable $exception
     * @param array $appends
     * @param string $type
     *
     * @param string $room
     * @author duc <1025434218@qq.com>
     */
    public function errorToUser(RpcContext $context, ?string $uuid, \Throwable $exception, array $appends = [], string $type = '', string $room = null)
    {
        $type = $this->parseType($context, $type);

        $this->shouldNotAddReturn($context, $type);

        $errorCode = $exception->getCode();
        $errorMsg = $exception->getMessage();

        $msg = json_encode([
            'success' => false,
            'code'    => $errorCode,
            'data'    => $appends,
            'msg'     => $errorMsg,
            'type'    => $type,
        ], JSON_UNESCAPED_UNICODE);

        $this->send($context, $uuid, $msg, $room);
    }

    /**
     * @param RpcContext $context
     * @param string $notifyType
     * @param array $payload
     * @param string $room
     *
     * @author duc <1025434218@qq.com>
     */
    public function notifyToSelf(RpcContext $context, string $notifyType, array $payload = [], string $room = null)
    {
        $msg = json_encode([
            'success' => true,
            'data'    => [
                'notify_type' => $notifyType,
                'payload'     => $payload,
            ],
            'type'    => self::WS_NOTIFY,
        ], JSON_UNESCAPED_UNICODE);

        $this->send($context, $context->getUserId(), $msg, $room);
    }

    /**
     * @param RpcContext $context
     * @param string $notifyType
     * @param string|null $uuid
     * @param array $payload
     * @param string $room
     *
     * @author duc <1025434218@qq.com>
     */
    public function notifyUser(RpcContext $context, string $notifyType, ?string $uuid = null, array $payload = [], string $room = null)
    {
        $msg = json_encode([
            'success' => true,
            'data'    => [
                'notify_type' => $notifyType,
                'payload'     => $payload,
            ],
            'type'    => self::WS_NOTIFY,
        ], JSON_UNESCAPED_UNICODE);

        $uuid = $uuid ?? $context->getUserId();

        $this->send($context, $uuid, $msg, $room);
    }

    /**
     * @param RpcContext $context
     * @param string $notifyType
     * @param array $payload
     * @param string $room
     *
     * @author duc <1025434218@qq.com>
     */
    public function notifyAll(RpcContext $context, string $notifyType, array $payload = [], string $room = null)
    {
        $msg = json_encode([
            'success' => true,
            'data'    => [
                'notify_type' => $notifyType,
                'payload'     => $payload,
            ],
            'type'    => self::WS_NOTIFY,
        ], JSON_UNESCAPED_UNICODE);

        $this->send($context, '[all]', $msg, $room);
    }

    /**
     * @param RpcContext $context
     * @param string $type
     *
     * @author duc <1025434218@qq.com>
     */
    public function shouldNotAddReturn(RpcContext $context, string $type)
    {
        $type = $this->parseType($context, $type);
        $context->shouldAddReturn[$type] = false;
    }

    /**
     * @param RpcContext $context
     *
     * @author duc <1025434218@qq.com>
     */
    public function resetShouldAddReturn(RpcContext $context)
    {
        $context->shouldAddReturn = [];
    }

    /**
     * @param RpcContext $context
     * @param string $type
     * @return bool
     *
     * @author duc <1025434218@qq.com>
     */
    public function shouldAddReturn(RpcContext $context, string $type)
    {
        $type = $this->parseType($context, $type);

        return $context->shouldAddReturn[$type] ?? true;
    }

    /**
     * @param RpcContext $context
     * @param string $type
     * @return string
     *
     * @author duc <1025434218@qq.com>
     */
    public function parseType(RpcContext $context, string $type = ''): string
    {
        $type = $type ?: ($context->websocketActionType ?? '');

        if ($type) {
            return $type;
        }

        throw new RpcRuntimeException('unknown websocket type!');
    }

    /**
     * @param RpcContext $context
     * @param string|null $clientId
     * @param $msg
     * @param string|null $room
     *
     * @author duc <1025434218@qq.com>
     */
    protected function send(RpcContext $context, ?string $clientId, $msg, ?string $room)
    {
        $roomName = $room ?: $context->getRoomName();
        if ($roomName && $clientId) {
            if ($this->shouldQueued()) {
                $this->push($context, $clientId, $roomName, $msg);

                return;
            }

            $this->sendToUser($clientId, $roomName, $msg);
        }
    }

    /**
     * @param string|null $uuid
     * @param string $room
     * @param string $msg
     *
     * @author duc <1025434218@qq.com>
     */
    public function sendToUser(?string $uuid, string $room, string $msg)
    {
        app(RpcMessageQueue::class)->addMessageToWebSocket($room, $uuid, $msg);
    }

    public function push(RpcContext $context, string $clientId, string $roomName, ?string $msg)
    {
        $contextKey = config('ws.ws_queued_data_key', 'ws_queued_data');
        $oldData = $context->{$contextKey} ?? [];
        $key = $roomName . '|' . $clientId;
        $data = [
            $key => [$msg],
        ];

        $context->{$contextKey} = array_merge_recursive($oldData, $data);
    }

    public function fire(RpcContext $context)
    {
        $items = $context->{config('ws.ws_queued_data_key', 'ws_queued_data')};
        foreach ($items as $key => $messages) {
            [$room, $clientId] = explode('|', $key);

            $this->sendToUser($clientId, $room, json_encode($messages));
        }
    }

    public function shouldQueued()
    {
        return ! ! config('ws.queued', false);
    }
}
