<?php

namespace Mainto\Websocket\Router;

use Mainto\Websocket\Pipeline;
use Illuminate\Support\Arr;
use Mainto\Websocket\Traits\RouteDependencyResolverTrait;

/**
 * Class Route
 * @package App\Services
 */
class Route
{
    use RouteDependencyResolverTrait;

    /**
     * @var string
     */
    protected $method = 'process';

    /**
     * @var string
     */
    protected $room;

    /**
     * @var array
     */
    protected $middleware = [];

    /**
     * @var Router
     */
    protected $router;

    /**
     * @var \Closure|string
     */
    private $handler;

    /**
     * Route constructor.
     * @param string $room
     * @param string $type
     * @param $handler
     * @param array $parentGroupMiddleware
     */
    public function __construct(string $room, string $type, $handler, $parentGroupMiddleware = [])
    {
        $this->type = $type;
        $this->room = $room;
        $this->handler = $handler;

        $this->middleware = collect($parentGroupMiddleware)->flatten()->toArray();
    }

    /**
     * @return mixed
     * @author duc <1025434218@qq.com>
     */
    public function dispatch()
    {
        return (new Pipeline())
            ->send($this->router->request, ...$this->router->payload)
            ->through($this->middleware)
            ->then(function ($passable) {
                return $this->handle($passable);
            });
    }

    /**
     * @param $passable
     * @return mixed
     * @throws \ReflectionException
     * @author duc <1025434218@qq.com>
     */
    private function handle($passable)
    {
        $payload = array_values(Arr::except($passable, 0));

        if (is_callable($this->handler)) {
            return $this->runCallable($passable, $payload);
        }

        return $this->runController($passable, $payload);
    }

    /**
     * @param array|string|null $middleware
     * @return $this
     *
     * @author duc <1025434218@qq.com>
     */
    public function middleware($middleware = null)
    {
        $middleware = is_array($middleware) ? $middleware : func_get_args();

        $this->middleware = array_merge($this->middleware, $middleware);

        return $this;
    }

    /**
     * @param mixed $router
     * @return Route
     */
    public function setRouter($router)
    {
        $this->router = $router;

        return $this;
    }

    /**
     * @param string|null $method
     * @return $this
     *
     * @author duc <1025434218@qq.com>
     */
    public function method(?string $method)
    {
        if ($method) {
            $this->method = $method;
        }

        return $this;
    }

    /**
     * @return array
     */
    public function getMiddleware(): array
    {
        return $this->middleware;
    }

    /**
     * @param $passable
     * @param array $payload
     * @return mixed
     *
     * @throws \ReflectionException
     * @author 神符 <1025434218@qq.com>
     */
    private function runController ($passable, array $payload) {
        $context = $passable[0];
        $instance = app($this->handler, ['context' => $context, 'payload' => $payload]);

        $parameters = $this->resolveClassMethodDependencies(
            $context, ['context' => $context, 'payload' => $payload], $instance, $this->method
        );

        return call_user_func_array([$instance, $this->method], $parameters);
    }

    /**
     * @param $passable
     * @param array $payload
     * @return mixed
     * @throws \ReflectionException
     *
     * @author 神符 <1025434218@qq.com>
     */
    private function runCallable ($passable, array $payload) {
        return ($this->handler)(...array_values($this->resolveMethodDependencies(
            $passable[0], ['context' => $passable[0], 'payload' => $payload], new \ReflectionFunction($this->handler)
        )));
    }
}
