<?php

namespace Mainto\Websocket;

use Mainto\RpcServer\RpcServer\RpcContext;
use Mainto\RpcServer\RpcUtil\RpcMessageQueue;
use Mainto\RpcServer\Exceptions\RpcRuntimeException;
use Mainto\Websocket\Contracts\Response as ResponseImp;

/**
 * Class Response
 * @package Mainto\Websocket
 */
class Response implements ResponseImp
{
    const WS_NOTIFY = 'notify';

    /**
     * @param RpcContext $context
     * @param $data
     * @param string $type
     *
     * @param string $room
     * @author duc <1025434218@qq.com>
     */
    public function successBroadcast(RpcContext $context, $data, string $type = '', string $room = null)
    {
        $type = $this->parseType($context, $type);

        $this->shouldNotAddReturn($context, $type);
        $msg = json_encode([
            'success' => true,
            'data'    => $data,
            'type'    => $type,
        ], JSON_UNESCAPED_UNICODE);

        $this->send($context, '[all]', $msg, $room);
    }

    /**
     * @param RpcContext $context
     * @param $data
     * @param string $type
     *
     * @param string $room
     * @author duc <1025434218@qq.com>
     */
    public function successToSelf(RpcContext $context, $data, string $type = '', string $room = null)
    {
        $type = $this->parseType($context, $type);

        $this->shouldNotAddReturn($context, $type);

        $msg = json_encode([
            'success' => true,
            'data'    => $data,
            'type'    => $type,
        ], JSON_UNESCAPED_UNICODE);

        $this->send($context, $context->getUserId(), $msg, $room);
    }

    /**
     * @param RpcContext $context
     * @param string $uuid
     * @param $data
     * @param string $type
     *
     * @param string $room
     * @author duc <1025434218@qq.com>
     */
    public function successToUser(RpcContext $context, string $uuid, $data, string $type = '', string $room = null)
    {
        $type = $this->parseType($context, $type);

        $this->shouldNotAddReturn($context, $type);

        $msg = json_encode([
            'success' => true,
            'data'    => $data,
            'type'    => $type,
        ], JSON_UNESCAPED_UNICODE);

        $this->send($context, $uuid, $msg, $room);
    }

    /**
     * @param RpcContext $context
     * @param \Exception $exception
     * @param array $appends
     * @param string $type
     *
     * @param string $room
     * @author duc <1025434218@qq.com>
     */
    public function errorBroadcast(RpcContext $context, \Exception $exception, array $appends = [], string $type = '', string $room = null)
    {
        $type = $this->parseType($context, $type);

        $this->shouldNotAddReturn($context, $type);
        $errorCode = $exception->getCode();
        $errorMsg = $exception->getMessage();

        $msg = json_encode([
            'success' => false,
            'code'    => $errorCode,
            'data'    => $appends,
            'msg'     => $errorMsg,
            'type'    => $type,
        ], JSON_UNESCAPED_UNICODE);

        $this->send($context, '[all]', $msg, $room);
    }

    /**
     * @param RpcContext $context
     * @param \Exception $exception
     * @param array $appends
     * @param string $type
     *
     * @param string $room
     * @author duc <1025434218@qq.com>
     */
    public function errorToSelf(RpcContext $context, \Exception $exception, array $appends = [], string $type = '', string $room = null)
    {
        $type = $this->parseType($context, $type);

        $this->shouldNotAddReturn($context, $type);
        $errorCode = $exception->getCode();
        $errorMsg = $exception->getMessage();

        $msg = json_encode([
            'success' => false,
            'code'    => $errorCode,
            'data'    => $appends,
            'msg'     => $errorMsg,
            'type'    => $type,
        ], JSON_UNESCAPED_UNICODE);

        $this->send($context, $context->getUserId(), $msg, $room);
    }

    /**
     * @param RpcContext $context
     * @param string $uuid
     * @param \Exception $exception
     * @param array $appends
     * @param string $type
     *
     * @param string $room
     * @author duc <1025434218@qq.com>
     */
    public function errorToUser(RpcContext $context, string $uuid, \Exception $exception, array $appends = [], string $type = '', string $room = null)
    {
        $type = $this->parseType($context, $type);

        $this->shouldNotAddReturn($context, $type);

        $errorCode = $exception->getCode();
        $errorMsg = $exception->getMessage();

        $msg = json_encode([
            'success' => false,
            'code'    => $errorCode,
            'data'    => $appends,
            'msg'     => $errorMsg,
            'type'    => $type,
        ], JSON_UNESCAPED_UNICODE);

        $this->send($context, $uuid, $msg, $room);
    }

    /**
     * @param RpcContext $context
     * @param string $notifyType
     * @param array $payload
     * @param string $room
     *
     * @author duc <1025434218@qq.com>
     */
    public function notifyToSelf(RpcContext $context, string $notifyType, array $payload = [], string $room = null)
    {
        $msg = json_encode([
            'success' => true,
            'data'    => [
                'notify_type' => $notifyType,
                'payload'     => $payload,
            ],
            'type'    => self::WS_NOTIFY,
        ], JSON_UNESCAPED_UNICODE);

        $this->send($context, $context->getUserId(), $msg, $room);
    }

    /**
     * @param RpcContext $context
     * @param string $notifyType
     * @param string|null $uuid
     * @param array $payload
     * @param string $room
     *
     * @author duc <1025434218@qq.com>
     */
    public function notifyUser(RpcContext $context, string $notifyType, string $uuid = null, array $payload = [], string $room = null)
    {
        $msg = json_encode([
            'success' => true,
            'data'    => [
                'notify_type' => $notifyType,
                'payload'     => $payload,
            ],
            'type'    => self::WS_NOTIFY,
        ], JSON_UNESCAPED_UNICODE);

        $uuid = $uuid ?? $context->getUserId();

        $this->send($context, $uuid, $msg, $room);
    }

    /**
     * @param RpcContext $context
     * @param string $notifyType
     * @param array $payload
     * @param string $room
     *
     * @author duc <1025434218@qq.com>
     */
    public function notifyAll(RpcContext $context, string $notifyType, array $payload = [], string $room = null)
    {
        $msg = json_encode([
            'success' => true,
            'data'    => [
                'notify_type' => $notifyType,
                'payload'     => $payload,
            ],
            'type'    => self::WS_NOTIFY,
        ], JSON_UNESCAPED_UNICODE);

        $this->send($context, '[all]', $msg, $room);
    }

    /**
     * @param RpcContext $context
     * @param string $type
     *
     * @author duc <1025434218@qq.com>
     */
    public function shouldNotAddReturn(RpcContext $context, string $type)
    {
        $type = $this->parseType($context, $type);
        $context->shouldAddReturn[$type] = false;
    }

    /**
     * @param RpcContext $context
     *
     * @author duc <1025434218@qq.com>
     */
    public function resetShouldAddReturn(RpcContext $context)
    {
        $context->shouldAddReturn = [];
    }

    /**
     * @param RpcContext $context
     * @param string $type
     * @return bool
     *
     * @author duc <1025434218@qq.com>
     */
    public function shouldAddReturn(RpcContext $context, string $type)
    {
        $type = $this->parseType($context, $type);

        return $context->shouldAddReturn[$type] ?? true;
    }

    /**
     * @param RpcContext $context
     * @param string $type
     * @return string
     *
     * @author duc <1025434218@qq.com>
     */
    public function parseType(RpcContext $context, string $type = ''): string
    {
        $type = $type ?: ($context->websocketActionType ?? '');

        if ($type) {
            return $type;
        }

        throw new RpcRuntimeException('unknown websocket type!');
    }

    /**
     * @param RpcContext $context
     * @param string $clientId
     * @param $msg
     * @param string $room
     *
     * @author duc <1025434218@qq.com>
     */
    protected function send(RpcContext $context, ?string $clientId, $msg, ?string $room)
    {
        $roomName = $context->getRoomName() ?? $room;
        if ($roomName && $clientId) {
            $this->sendToUser($clientId, $roomName, $msg);
        }
    }

    /**
     * @param string $uuid
     * @param string $room
     * @param string $msg
     *
     * @author duc <1025434218@qq.com>
     */
    public function sendToUser(string $uuid, string $room, string $msg)
    {
        app(RpcMessageQueue::class)->addMessageToWebSocket($room, $uuid, $msg);
    }
}
