<?php

namespace Mainto\Websocket\Auth;

use Mainto\Websocket\Contracts\Userable;
use Mainto\RpcServer\RpcServer\RpcContext;
use Illuminate\Validation\UnauthorizedException;
use Mainto\RpcServer\Exceptions\RpcRuntimeException;

/**
 * Class Manager
 * @package Mainto\Websocket\Auth
 */
class Manager
{
    /**
     * @var RpcContext
     */
    public $context;

    /**
     * @var UserProvider
     */
    protected $provider;

    /**
     * AuthGuard constructor.
     * @param RpcContext $context
     * @throws \Exception
     */
    public function __construct(RpcContext $context)
    {
        $this->context = $context;
        $this->setWsUserProvider();
    }

    /**
     * Log a user into the application.
     *
     * @param Userable $user
     * @return bool
     */
    public function login(Userable $user)
    {
        $user->setUuid($this->context->getUserId());

        if (! $user->getKey()) {
            return false;
        }

        $this->getProvider()->login($user);
        $this->setWebsocketUser($user);

        return true;
    }

    /**
     * Log the user out of the application.
     *
     * @return void
     */
    public function logout()
    {
        $this->getProvider()->logout($this->user());
        $this->setWebsocketUser(null);
    }

    /**
     * @return Userable|null
     *
     * @author duc <1025434218@qq.com>
     */
    public function user()
    {
        $user = $this->getWebsocketUser();

        if ($user) {
            if ($this->getProvider()->isLogin($user)) {
                return $user;
            }

            throw new UnauthorizedException();
        }

        if (! $user) {
            $user = $this->setUserByUuid($this->context->getUserId());
        }

        return $user;
    }

    /**
     * Determine if the current user is a guest.
     *
     * @return bool
     */
    public function guest()
    {
        $uuid = $this->context->getUserId();

        $user = $this->getProvider()->getUserByUuid($uuid);

        if ($user && $user->getUuid() == $this->getProvider()->getUuidByKey($user->getKey())) {
            $this->login($user);

            return false;
        }

        $this->getProvider()->dissociate($uuid);

        return true;
    }

    /**
     * Determine if the current user is authenticated.
     *
     * @return bool
     */
    public function check()
    {
        return ! $this->guest();
    }

    /**
     * @return bool
     *
     * @author duc <1025434218@qq.com>
     */
    public function hasUser()
    {
        return ! ! $this->user();
    }

    /**
     * Get the ID for the currently authenticated user.
     *
     * @return string|null
     * @throws \Exception
     */
    public function uuid()
    {
        return $this->user()
            ? $this->user()->getUuid()
            : $this->context->getUserId();
    }

    /**
     * Set the current user.
     *
     * @param string $uuid
     * @return Userable|null
     */
    protected function setUserByUuid(string $uuid)
    {
        $user = $this->getProvider()->getUserByUuid($uuid);

        if (! $user) {
            return null;
        }

        $this->login($user);

        return $user;
    }

    /**
     * @return UserProvider
     *
     * @author duc <1025434218@qq.com>
     */
    public function getProvider()
    {
        return $this->provider;
    }

    /**
     * @return Userable|null
     *
     * @author duc <1025434218@qq.com>
     */
    public function getWebsocketUser()
    {
        return $this->context->websocketUser ?? null;
    }

    /**
     * @param Userable|null $user
     *
     * @author duc <1025434218@qq.com>
     */
    protected function setWebsocketUser(?Userable $user): void
    {
        $this->context->websocketUser = $user;
    }

    /**
     * @throws \Exception
     *
     * @author duc <1025434218@qq.com>
     */
    protected function setWsUserProvider(): void
    {
        $this->provider = app(UserProvider::class, ['auth' => $this]);
    }

    /**
     * @return string|null
     *
     * @author duc <1025434218@qq.com>
     */
    public function room()
    {
        if ($room = $this->context->getRoomName()) {
            return $room;
        }

        throw new RpcRuntimeException();
    }
}
