<?php

namespace Mainto\Websocket;

use Closure;
use Illuminate\Support\Arr;

/**
 * Class Pipeline
 * @package Mainto\Websocket
 */
class Pipeline
{
    /**
     * The object being passed through the pipeline.
     *
     * @var array
     */
    protected $passable = [];

    /**
     * @var int
     */
    protected $passableNum;

    /**
     * @var bool
     */
    protected $firstArgIsArray = false;

    /**
     * The array of class pipes.
     *
     * @var array
     */
    protected $pipes = [];

    /**
     * The method to call on each pipe.
     *
     * @var string
     */
    protected $method = 'handle';

    /**
     * Set the object being sent through the pipeline.
     *
     * @param mixed $passable
     * @return Pipeline
     */
    public function send($passable)
    {
        $this->passable = func_get_args();
        $this->passableNum = count($this->passable);
        $this->firstArgIsArray = is_array($this->passable[0]);

        return $this;
    }

    /**
     * Set the array of pipes.
     *
     * @param array|mixed $pipes
     * @return $this
     */
    public function through($pipes)
    {
        $this->pipes = is_array($pipes) ? $pipes : func_get_args();

        return $this;
    }

    /**
     * Set the method to call on the pipes.
     *
     * @param string $method
     * @return $this
     */
    public function via($method)
    {
        $this->method = $method;

        return $this;
    }

    /**
     * Run the pipeline with a final destination callback.
     *
     * @param \Closure $destination
     * @return mixed
     */
    public function then(Closure $destination)
    {
        $pipeline = array_reduce(
            array_reverse($this->pipes),
            $this->carry(),
            $this->prepareDestination($destination)
        );

        return $pipeline(...$this->passable);
    }

    /**
     * Get the final piece of the Closure onion.
     *
     * @param \Closure $destination
     * @return \Closure
     */
    protected function prepareDestination(Closure $destination)
    {
        return function ($passable) use ($destination) {
            $args = func_get_args();
            if (count($args) >= 1 && $this->passableNum > 1) {
                $passable = $args;
            } else {
                $passable = is_array($passable) && ! $this->firstArgIsArray ? $passable : [$passable];
            }

            return $destination($passable);
        };
    }

    /**
     * Get a Closure that represents a slice of the application onion.
     *
     * @return \Closure
     */
    protected function carry()
    {
        return function ($stack, $pipe) {
            return function ($passable = []) use ($stack, $pipe) {
                $args = func_get_args();
                if (count($args) >= 1 && $this->passableNum > 1) {
                    $passable = $args;
                } else {
                    $passable = is_array($passable) && ! $this->firstArgIsArray ? $passable : [$passable];
                }

                $payload = array_values(Arr::except($passable, 0));

                if (is_callable($pipe)) {
                    return $pipe($stack, $passable[0] ?? null, $payload);
                }

                $obj = app($pipe);
                $response = method_exists($obj, $this->method)
                    ? $obj->{$this->method}($stack, $passable[0] ?? null, $payload)
                    : $obj($stack, $passable[0] ?? null, $payload);

                return $response;
            };
        };
    }
}
