<?php

namespace Mainto\Websocket;

use Mainto\Websocket\Router\Route;
use Mainto\Websocket\Router\Router;
use Mainto\Websocket\Contracts\Response;
use Mainto\RpcServer\RpcServer\RpcContext;
use Mainto\Websocket\Middleware\Authenticate;
use Illuminate\Validation\UnauthorizedException;

/**
 * Class WsServer
 * @package App\Rpc\Service\NoPaper
 */
class WebsocketServer
{
    /**
     * @var array
     */
    protected $payload;

    /**
     * @var string
     */
    protected $room;

    /**
     * @var array
     */
    protected $middleware = [Authenticate::class];

    /**
     * @var Router
     */
    protected $router;

    /**
     * @var Response
     */
    protected $response;

    /**
     * WsServer constructor.
     * @param $room
     * @param $payload
     */
    public function __construct($room, ...$payload)
    {
        $this->room = $room;
        $this->payload = $payload;
        $this->router = app(Router::class);
        $this->response = app(Response::class);
    }

    /**
     * @param RpcContext $context
     * @param $type
     * @return mixed
     *
     * @author duc <1025434218@qq.com>
     */
    public function dispatch(RpcContext $context, $type)
    {
        $context->websocketActionType = $type;

        $this->response->resetShouldAddReturn($context);

        try {
            $response = $this->sendRequestThroughRouter($context, $type);

            if ($this->response->shouldAddReturn($context, $type)) {
                $this->response->successToSelf($context, []);
            }
        } catch (\Exception $e) {
            if ($e instanceof UnauthorizedException) {
                (function ($room = null) {
                    $this->code = config('ws.unauthorized.code.' . $room) ?? config('ws.unauthorized.code.default', 401);
                    $this->message = config('ws.unauthorized.message.' . $room) ?? config('ws.unauthorized.message.default', 'unauthorized!');
                })->bindTo($e, UnauthorizedException::class)($this->room);
            }

            if ($this->response->shouldAddReturn($context, $type)) {
                $this->response->errorToSelf($context, $e);
            }

            return $e;
        }

        /** @var Route $route */
        $route = $context->currentWebsocketRoute;

        foreach ($route->getMiddleware() as $middleware) {
            if (is_callable($middleware)) {
                continue;
            }

            $instance = app($middleware);
            if (method_exists($instance, 'terminate')) {
                $instance->terminate($context, $response);
            }
        }

        return $response;
    }

    /**
     * @param $middleware
     * @return $this
     *
     * @author duc <1025434218@qq.com>
     */
    public function addGlobalMiddleware($middleware)
    {
        $middleware = is_array($middleware) ? $middleware : func_get_args();
        $this->middleware = array_merge($this->middleware, $middleware);

        return $this;
    }

    /**
     * @param RpcContext $request
     * @param string $type
     * @return mixed
     *
     * @author duc <1025434218@qq.com>
     */
    protected function sendRequestThroughRouter($request, string $type)
    {
        $response = (new Pipeline())
            ->send($request, ...$this->payload)
            ->through($this->middleware ?? [])
            ->then(function ($passable) use ($request, $type) {
                /** @var Route $route */
                $route = $this->router
                    ->setRequest($request)
                    ->setPayload(...$this->payload)
                    ->findRoute($this->room, $type);
                $request->currentWebsocketRoute = $route;

                return $route->dispatch();
            });

        return $response;
    }
}
