<?php

namespace DucCnzj\WsPhp\Router;

use DucCnzj\WsPhp\Pipeline;
use Illuminate\Support\Arr;

/**
 * Class Route
 * @package App\Services
 */
class Route
{
    /**
     * @var string
     */
    protected $method = 'process';

    /**
     * @var string
     */
    protected $room;

    /**
     * @var array
     */
    protected $middleware = [];

    /**
     * @var Router
     */
    protected $router;

    /**
     * @var \Closure|string
     */
    private $handler;

    /**
     * Route constructor.
     * @param string $room
     * @param string $type
     * @param $handler
     * @param array $parentGroupMiddleware
     */
    public function __construct(string $room, string $type, $handler, $parentGroupMiddleware = [])
    {
        $this->type = $type;
        $this->room = $room;
        $this->handler = $handler;

        $this->middleware = collect($parentGroupMiddleware)->flatten()->toArray();
    }

    /**
     * @return mixed
     * @author duc <1025434218@qq.com>
     */
    public function dispatch()
    {
        return (new Pipeline())
            ->send($this->router->request, ...$this->router->payload)
            ->through($this->middleware)
            ->then(function ($passable) {
                return $this->handle($passable);
            });
    }

    /**
     * @param $passable
     * @return mixed
     * @author duc <1025434218@qq.com>
     */
    private function handle($passable)
    {
        $payload = array_values(Arr::except($passable, 0));
        if (is_callable($this->handler)) {
            return ($this->handler)($passable[0], $payload);
        }

        return (new $this->handler($passable[0], $payload))->{$this->method}($passable[0], $payload);
    }

    /**
     * @param array|string|null $middleware
     * @return $this
     *
     * @author duc <1025434218@qq.com>
     */
    public function middleware($middleware = null)
    {
        $middleware = is_array($middleware) ? $middleware : func_get_args();

        $this->middleware = array_merge($this->middleware, $middleware);

        return $this;
    }

    /**
     * @param mixed $router
     * @return Route
     */
    public function setRouter($router)
    {
        $this->router = $router;

        return $this;
    }

    /**
     * @param string|null $method
     * @return $this
     *
     * @author duc <1025434218@qq.com>
     */
    public function method(?string $method)
    {
        if ($method) {
            $this->method = $method;
        }

        return $this;
    }

    /**
     * @return array
     */
    public function getMiddleware(): array
    {
        return $this->middleware;
    }
}
