<?php

namespace DucCnzj\WsPhp\Auth;

use Carbon\Carbon;
use DucCnzj\WsPhp\Contracts\Userable;
use Illuminate\Support\Facades\Redis;
use DucCnzj\WsPhp\Events\ClearLastLoginInfo;

/**
 * Class UserProvider
 * @package DucCnzj\WsPhp\Auth
 */
class UserProvider
{
    /**
     * @var \Illuminate\Redis\Connections\Connection
     */
    protected $redis;

    /**
     * @var string
     */
    protected $keyUuidCacheKey;

    /**
     * @var string
     */
    protected $uuidUserCacheKey;

    /**
     * @var Manager
     */
    protected $auth;

    /**
     * WsUserProvider constructor.
     * @param Manager $auth
     * @throws \Exception
     */
    public function __construct(Manager $auth)
    {
        $this->auth = $auth;
        $this->redis = Redis::connection(config('ws.connection', 'default'));
        $this->uuidUserCacheKey = sprintf('%s:%s', $this->auth->room(), 'uuid_cache_key');
        $this->keyUuidCacheKey = sprintf('%s:%s', $this->auth->room(), 'all_login_users');
    }

    /**
     * @param Userable $user
     *
     * @author duc <1025434218@qq.com>
     */
    public function login(Userable $user)
    {
        $key = $this->getKey($user);

        if (($oldUuid = $this->redis->hget($this->keyUuidCacheKey, $key)) && $oldUuid != $user->getUuid()) {
            $oldUserInfo = $this->getUserByUuid($oldUuid);
            $this->redis->hdel($this->uuidUserCacheKey, [$oldUuid]);
            $this->redis->hdel($this->keyUuidCacheKey, [$key]);
            event(new ClearLastLoginInfo($oldUserInfo, $this->auth->room()));
        }

        if ($user->getUuid() && $this->getKey($user)) {
            $this->cacheLoginUserUuid($user);
            $this->cacheUuidUserMap($user);
        }
    }

    /**
     * @param Userable $user
     * @return bool
     *
     * @author duc <1025434218@qq.com>
     */
    public function isLogin(Userable $user)
    {
        return in_array($user->getUuid(), array_keys($this->getAllUuidUsers()));
    }

    /**
     * @param Userable|null $user
     *
     * @author duc <1025434218@qq.com>
     */
    public function logout(?Userable $user)
    {
        if ($user) {
            $this->redis->hdel($this->keyUuidCacheKey, [$this->getKey($user)]);
            $this->redis->hdel($this->uuidUserCacheKey, [$user->getUuid()]);
        }
    }

    /**
     * @param string $uuid
     * @return User|null
     *
     * @author duc <1025434218@qq.com>
     */
    public function getUserByUuid(string $uuid)
    {
        return @unserialize($this->redis->hget($this->uuidUserCacheKey, $uuid)) ?: null;
    }

    /**
     * @param $key
     * @return string
     *
     * @author duc <1025434218@qq.com>
     */
    public function getUuidByKey($key)
    {
        return $this->redis->hget($this->keyUuidCacheKey, $this->getKey($key));
    }

    /**
     * @param Userable $user
     *
     * @author duc <1025434218@qq.com>
     */
    protected function cacheLoginUserUuid(Userable $user): void
    {
        $this->setExpireIfNotExist($this->keyUuidCacheKey, function ($redis) use ($user) {
            $redis->hset($this->keyUuidCacheKey, $this->getKey($user), $user->getUuid());
        });
    }

    /**
     * @param string $uuid
     *
     * @author duc <1025434218@qq.com>
     */
    public function dissociate(string $uuid)
    {
        $this->redis->hdel($this->uuidUserCacheKey, [$uuid]);
    }

    /**
     * @param Userable $user
     *
     * @author duc <1025434218@qq.com>
     */
    protected function cacheUuidUserMap(Userable $user): void
    {
        $this->setExpireIfNotExist($this->uuidUserCacheKey, function ($redis) use ($user) {
            $redis->hset($this->uuidUserCacheKey, $user->getUuid(), serialize($user));
        });
    }

    /**
     * @param string $key
     * @param callable $callback
     * @param int|null $seconds
     * @return mixed
     *
     * @author duc <1025434218@qq.com>
     */
    public function setExpireIfNotExist(string $key, callable $callback, int $seconds = null)
    {
        if (! $this->redis->exists($key)) {
            $res = $callback($this->redis);
            $this->redis->expire($key, $seconds ?? $this->getCacheSeconds());
        } else {
            $res = $callback($this->redis);
        }

        return $res;
    }

    /**
     * @return int
     *
     * @author duc <1025434218@qq.com>
     */
    private function getCacheSeconds()
    {
        return now()->diffInRealSeconds(Carbon::tomorrow());
    }

    /**
     * @return \Illuminate\Redis\Connections\Connection
     *
     * @author duc <1025434218@qq.com>
     */
    public function getRedis()
    {
        return $this->redis;
    }

    /**
     * @return array
     *
     * @author duc <1025434218@qq.com>
     */
    public function getAllKeyUuid()
    {
        return $this->redis->hgetall($this->keyUuidCacheKey);
    }

    /**
     * @return array
     *
     * @author duc <1025434218@qq.com>
     */
    public function getAllUuidUsers()
    {
        return $this->redis->hgetall($this->uuidUserCacheKey);
    }

    /**
     * @param $key
     * @return string
     *
     * @author duc <1025434218@qq.com>
     */
    protected function getKey($key)
    {
        if ($key instanceof Userable) {
            $key = $key->getKey();
        }

        return sprintf('user:%s', $key);
    }
}
