<?php

namespace App\Infrastructure\Console\Helper;

use Illuminate\Support\Str;
use phpDocumentor\Reflection\DocBlock\Tags\Property;
use phpDocumentor\Reflection\DocBlockFactory;
use phpDocumentor\Reflection\Type;
use phpDocumentor\Reflection\Types\Compound;

class Reflection {
    private $refclass = null;

    /**
     * 设置反射类
     * @param $classname
     * @return Reflection
     * @throws \ReflectionException
     */
    public function setClass ($classname): Reflection {
        $this->refclass = new \ReflectionClass($classname);
        return $this;
    }

    public function functionExist (string $functionName) {
        $methods = $this->refclass->getMethods(\ReflectionMethod::IS_PUBLIC);
        return collect($methods)->where('name', $functionName)->count() > 0;
    }


    public function endLine () {
        return $this->refclass->getEndLine();
    }

    /**
     * 获取类的文档信息
     * @param string $docTagName
     * @return array
     */
    public function getDocComment ($docTagName = 'property'): array {
        if (!$docComment = $this->refclass->getDocComment()) {
            return [];
        }
        $comment = [];
        // 使用 ReflectionDocBlock
        $factory = DocBlockFactory::createInstance();
        $docblock = $factory->create($docComment);
        foreach ($docblock->getTagsByName($docTagName) as $docComment) {
            $comment[] = $this->docBlockParser($docComment);
        }
        return $comment;
    }

    /**
     * 获取类的属性
     * @return array
     */
    public function getProperties (): array {
        if (!$properties = $this->refclass->getProperties()) {
            return [];
        }
        return collect($properties)->pluck('name')->toArray();
    }

    /**
     * 解析ReflectionDocBlock 的数据
     * 不使用反射，消耗性能
     * @param $docComment
     * @return array
     */
    protected function docBlockParser ($docComment): array {
        $closure = function () {
            $defaultNull = false;
            /**@var Property $this */
            $docTypeList = [];
            // todo 如果修改了model信息，显示会出错
            $docCommentType = $this->getType();
            if ($docCommentType instanceof Compound) {
                collect($docCommentType)->map(function (?Type $type) use (&$defaultNull, &$docTypeList) {
                    self::docType($type, $defaultNull, $docTypeList);
                });
            } else {
                self::docType($docCommentType, $defaultNull, $docTypeList);
            }

            $docTypeList = array_unique($docTypeList);
            $type = current($docTypeList);
            $docType = implode('|', $docTypeList);
            if ($defaultNull) {
                $docType .= '|null';
            }
            return [
                'variableName' => Str::camel($this->getVariableName()),
                'docType'      => $docType,
                'type'         => $type,
                'defaultNull'  => $defaultNull,
                'description'  => $this->getDescription()->getBodyTemplate(),
            ];
        };
        $method = $closure->bindTo($docComment, DocBlackParser::class);
        return $method();
    }


}

