<?php

namespace App\Infrastructure\Console\DomainWheel;

use App\Infrastructure\Console\Helper\Reflection;
use Psy\Exception\RuntimeException;

/**
 * Class MakeEntity
 * @package App\Console\DomainWheel
 * @property string $entityFolderFullPath                实体文件夹的完整路径
 * @property string $voFolderFullPath                    值对象的完整路径
 * @property string $entityClassNameSpace                领域的命名空间
 * @property string $voClassNameSpace                    值对象的命名空间
 * @property string $poFolderFullPath                    实体关联PO的完整路径
 */
class MakeEntity implements MakeWheel {
    use MakeWheelTools;

    protected string $domain;

    protected string $entity;

    protected string $entityPOFile;

    protected string $voName;

    protected string $entityStub = __DIR__."/Stubs/EntityClass.stub";

    protected string $voStub = __DIR__."/Stubs/VOClass.stub";


    protected MakeFolder $makeFolder;


    /**
     * MakeEntity constructor.
     * @param string $domain
     * @param string $entity
     */
    public function __construct (string $domain, string $entity) {
        $this->domain = $domain;
        $this->entity = $entity;
        $this->makeFolder = (new MakeFolder($this->domain));
        $this->validate();
    }

    /**
     * 设置实体文件夹完整路径的方法
     * @param string $entityFolderFullPath
     * @return $this
     */
    public function setEntityFolderFullPath (string $entityFolderFullPath): self {
        $this->entityFolderFullPath = $entityFolderFullPath;
        return $this;
    }

    /**
     * @param $property
     * @return mixed
     */
    public function __get ($property) {
        $method = 'get'.ucfirst($property);
        if (!method_exists($this, $method)) {
            throw new RuntimeException('该方法不存在');
        }
        return $this->$method();
    }

    /**
     * @return string
     */
    public function getEntityFolderFullPath (): string {
        return $this->makeFolder->getDomainEntityFolder();
    }

    /**
     * @return string
     */
    public function getVoFolderFullPath (): string {
        return $this->makeFolder->getDomainVOFullFolder();
    }

    /**
     * @return string
     */
    public function getEntityClassNameSpace (): string {
        return $this->domainEntityNameSpace($this->domain).'\\'.$this->entity.'Entity';
    }

    /**
     * @return string
     */
    public function getVoClassNameSpace (): string {
        return $this->domainVONameSpace($this->domain).'\\'.$this->voName;
    }


    /**
     * 校验PO信息
     * @todo 添加文件夹指定
     */
    private function validate () {
        if (!is_dir($this->voFolderFullPath)) {
            throw new RuntimeException('实体的所关联值对象不存在，请先创建相关文件夹');
        }

        $this->poFolderFullPath = $this->makeFolder->getEntityPoFullFolder();
        $this->entityPOFile = $this->poFolderFullPath.'/'.ucfirst($this->entity).'.php';
        if (!file_exists($this->entityPOFile)) {
            throw new RuntimeException('该持久对象不存在');
        }

    }

    /**
     * 1、反射获取model的数据
     * 2、生成实体类
     * 3、生成VO的方法
     * 4、model生成toEntity的方法
     */
    public function build () {
        // 创建VO 信息
        $this->buildVO();
        // 生成实体类
        $this->buildEntity();

        // build PO的toEntity
        $this->buildPOFunction();
    }

    /**
     * 创建值对象
     */
    private function buildVO () {
        $this->voName = $this->entity.'IdVO';
        $voFile = $this->voFolderFullPath.'/'.$this->voName.'.php';

        if (file_exists($voFile)) {
            return;
        }

        $stub = file_get_contents($this->voStub);


        $this->replaceNameSpace($stub, $this->domainVONameSpace($this->domain))
            ->replaceClass($stub, $this->voName)
            ->replaceProperty($stub, 'property', 'id');

        $this->put($voFile, $stub);
    }

    /**
     * 生成实体类
     */
    private function buildEntity () {
        $entityClass = $this->entity.'Entity';
        $entityFile = $this->entityFolderFullPath.'/'.$entityClass.'.php';
        // todo 反射获取每个变量，id需要设置VO
        if (file_exists($entityFile)) {
            return;
        }
        $poClass = $this->poNameSpace($this->entity);

        $stub = file_get_contents($this->entityStub);
        $this->replaceNameSpace($stub, $this->domainEntityNameSpace($this->domain))
            ->replaceClass($stub, $entityClass)
            ->replacePO($stub, $poClass);

        // 根据po创建方法
        $propertyList = app(Reflection::class)->setClass($poClass)->getDocComment();
        $properties = $voNameSpace = '';

        foreach ($propertyList ?? [] as $property) {
            if ($property['variableName'] == 'id') {
                $voNameSpace = 'use '.$this->voClassNameSpace.';';
                $property = [
                    'variableName' => $property['variableName'],
                    'docType'      => $this->voName.'|null',
                    'type'         => $this->voName,
                    'defaultNull'  => true,
                    'description'  => "id",
                ];
            }
            $properties .= $this->entityProperty($property);
        }

        $this->put($entityFile, sprintf($stub, $voNameSpace, $properties));
    }

    private function buildPOFunction () {
        // 判断是否存在当前类
        $poClass = $this->poNameSpace($this->entity);
        $poReflection = app(Reflection::class)->setClass($poClass);
        if ($poReflection->functionExist('toEntity') && $poReflection->functionExist('fromEntity')) {
            return;
        }

        // 持久对象的信息
        $poFile = file_get_contents($this->entityPOFile);


        // 查找第一个use，替换
        $useEntityString = 'use '.$this->entityClassNameSpace.';';
        if (strrpos($poFile, $useEntityString) === false) {
            $replaceStartLen = strpos($poFile, 'use ');
            $poFile = substr_replace($poFile, PHP_EOL.$useEntityString, $replaceStartLen - 1, 0);
        }

        $useVoString = 'use '.$this->voClassNameSpace.';';
        if (strrpos($poFile, $useVoString) === false) {
            $replaceStartLen = strpos($poFile, 'use ');
            $poFile = substr_replace($poFile, PHP_EOL.$useVoString, $replaceStartLen - 1, 0);
        }

        // 获取实体的所有变量
        $properties = app(Reflection::class)->setClass($this->entityClassNameSpace)->getProperties();
        $toEntityString = $fromEntityString = '';
        foreach ($properties ?? [] as $property) {
            if ($property == 'id') {
                continue;
            }
            $toEntityString .= $this->toEntityString($property);
            $fromEntityString .= $this->fromEntityString($property);
        }

        if (strrpos($poFile, 'toEntity') === false) {
            $poFileLength = strlen($poFile);
            $poFile = substr_replace($poFile, $this->toEntity($toEntityString), $poFileLength - 2, 0);
        }

        if (strrpos($poFile, 'fromEntity') === false) {
            $poFileLength = strlen($poFile);
            $poFile = substr_replace($poFile, $this->fromEntity($fromEntityString), $poFileLength - 2, 0);
        }


        $this->put($this->entityPOFile, $poFile);


    }

    // todo repository文件和方法  yaml （常见方法-->自定义use，方法及实现）


}