<?php


namespace App\Domain\ThirdPartyOrder\Service;


use App\Domain\ThirdPartyOrder\Exception\CMCCOrderException;
use GuzzleHttp\Exception\GuzzleException;
use Illuminate\Support\Facades\Http;
use Mainto\RpcServer\RpcUtil\Tool\RpcLog;
use App\Domain\ThirdPartyOrder\Cmd\CMCCSignCmd;
use App\Domain\ThirdPartyOrder\Enum\CMCCCodeEnum;
use App\Domain\ThirdPartyOrder\Cmd\CMCCOpenCallbackCmd;
use App\Infrastructure\Bridge\CMCC\Setting\CMCCSetting;
use App\Infrastructure\Bridge\DingDing\DingDingNoticeService;

/**
 * CMCC 相关请求服务
 * Class CMCCApiService
 * @package App\Domain\ThirdPartyOrder\Service
 */
class CMCCApiService {

    /**
     * @param CMCCSignCmd $signCmd
     * @param string $privateKey
     * @return bool
     */
    public function checkSign (CMCCSignCmd $signCmd, string $privateKey): bool {
        return $signCmd->sign == $this->makeSign($signCmd, $privateKey);
    }

    /**
     * 生成签名
     * @param CMCCSignCmd $signCmd
     * @param $privateKey
     * @param bool $isSort
     * @return string
     */
    public function makeSign (CMCCSignCmd $signCmd, $privateKey, $isSort = true): string {
        $body = array_filter(json_decode($signCmd->body, true), function (string $val) {
            return $val !== null;
        });

        if ($isSort) {
            ksort($body);
        }

        return md5($signCmd->transactionId.$signCmd->reqTime.$privateKey.json_encode($body, JSON_UNESCAPED_UNICODE));
    }

    /**
     * 对券码加密
     * @param string $coupon
     * @return string
     */
    public function encryptCoupon (string $coupon): string {
        return base64_encode(openssl_encrypt($coupon, 'AES-128-CBC', CMCCSetting::newInstance()->callbackPrivateKey, OPENSSL_RAW_DATA, str_repeat('z', 16)));
    }

    /**
     * @param CMCCOpenCallbackCmd $params
     * @param string $traceId
     * @return bool
     * @throws GuzzleException
     */
    public function openCallbackNotice (CMCCOpenCallbackCmd $params, string $traceId): bool {
        $res = $this->CMCCApiRequest('commonBusinessOpenCallback', $params->toArray());
        if (!empty($res['contractRoot']['body']['resultCode']) && $res['contractRoot']['body']['resultCode'] == CMCCCodeEnum::SUCCESS) {
            return true;
        }

        // 钉钉群报警通知
        DingDingNoticeService::sendDingDingErrorNotice('通知中国移动回调信息报错', $traceId);
        throw CMCCOrderException::CMCC_API_FAIL();
    }

    /**
     * @param string $url
     * @param array $params
     * @return array
     */
    public function CMCCApiRequest (string $url, array $params): array {
        $url = CMCCSetting::newInstance()->url.$url;
        try {
            $res = Http::withBody(json_encode($params, JSON_UNESCAPED_UNICODE), 'application/json')->post($url)->body();
            // 记录日志
            RpcLog::getInstance()->info('thirdProxy:cmcc_request_api', ['url' => $url, 'params' => $params, 'response' => $res]);

            return json_decode($res, true);
        } catch (\Exception $exception) {
            RpcLog::getInstance()->error('thirdProxy:cmcc_request_api_fail', [
                'error'  => $exception->getMessage(),
                'params' => $params,
                'url'    => $url,
            ]);

            return [];
        }
    }
}