<?php

namespace Mainto\RpcServer\RpcUtil\Tool\LogDriver;

use Illuminate\Support\Str;
use InvalidArgumentException;
use Mainto\RpcServer\RpcUtil\Tool\RpcLogInterface;
use Mainto\RpcServer\Protocol\Request\Request;
use Mainto\RpcServer\Tracing\SpanArray;
use Monolog\Logger as Monolog;
use Throwable;
use Closure;

abstract class LogDriveAbstract implements RpcLogInterface {

    /**
     * 默认日志level
     * @var int
     */
    protected int $level = Monolog::DEBUG;

    /**
     * The Log levels.
     *
     * @var array
     */
    protected array $levels = [
        'debug'     => Monolog::DEBUG,
        'info'      => Monolog::INFO,
        'notice'    => Monolog::NOTICE,
        'warning'   => Monolog::WARNING,
        'error'     => Monolog::ERROR,
        'critical'  => Monolog::CRITICAL,
        'alert'     => Monolog::ALERT,
        'emergency' => Monolog::EMERGENCY,
    ];


    public function __construct (?string $level) {
        $this->level = $this->level($level);
    }

    /**
     * Parse the string level into a Monolog constant.
     *
     * @param string $level
     * @return int
     *
     * @throws InvalidArgumentException
     */
    protected function level (string $level = 'debug'): int {
        $level = strtolower($level);

        if (isset($this->levels[$level])) {
            return $this->levels[$level];
        }

        throw new InvalidArgumentException('Invalid log level.');
    }

    /**
     * @param string $level
     * @return bool
     */
    protected function isHandling (string $level): bool {
        return $this->level($level) >= $this->level;
    }

    /**
     * @param $message
     * @return string
     */
    protected function formatMessage ($message): string {
        try {
            if ($message instanceof Closure) {
                $message = $message();
            }
            return json_encode($message, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        } catch (Throwable $e) {
            return $e->getMessage();
        }
    }


    /**
     * get trace info
     *
     * @param int $offset
     * @return array
     */
    protected function getTraceInfo (int $offset = 0): array {
        $trace = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS, 4);
        $file = $trace[2 + $offset]['file'] ?? "";
        $line = $trace[2 + $offset]['line'] ?? "unknown";
        $class = $trace[3 + $offset]['class'] ?? "";
        $type = $trace[3 + $offset]['type'] ?? "::";
        $function = $trace[3 + $offset]['function'] ?? "";

        if (Str::endsWith($function, "{closure}")) {
            $function = "{closure}";
        }

        return [
            "source" => "{$class}{$type}{$function} ($line)",
            "file"   => "{$file} ($line)",
        ];
    }


    /**
     * log throwable error log
     *
     * @param Request $request
     * @param Throwable $e
     * @param string $source
     */
    abstract public function logRequestThrowable (Request $request, Throwable $e, string $source = "");

    abstract public function logThrowable ($request, Throwable $e, string $source = "");

    /**
     * log info message
     *
     * @param string $title
     * @param $message
     * @return mixed
     */
    abstract public function info (string $title, $message);

    /**
     * log debug message
     *
     * @param string $title
     * @param $message
     * @return mixed
     */
    abstract public function debug (string $title, $message);

    /**
     * log notice message
     *
     * @param string $title
     * @param $message
     * @return mixed
     */
    abstract public function notice (string $title, $message);

    /**
     * log warning message
     *
     * @param string $title
     * @param $message
     * @return mixed
     */
    abstract public function warning (string $title, $message);

    /**
     * log error message
     *
     * @param string $title
     * @param $message
     * @return mixed
     */
    abstract public function error (string $title, $message);

    /**
     * log critical message
     *
     * @param string $title
     * @param $message
     * @return mixed
     */
    abstract public function critical (string $title, $message);

    /**
     * @param string $title
     * @param $message
     * @return mixed
     */
    abstract public function sendEvent (string $title, $message);

    /**
     * log critical message
     *
     * @param SpanArray $spans
     */
    abstract public function tracer (SpanArray $spans);
}