<?php

use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Contracts\Support\Jsonable;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Mainto\RpcServer\Exceptions\RpcRuntimeException;
use Mainto\RpcServer\RpcClient\RpcClientFactory;
use Mainto\RpcServer\RpcServer\RpcEnvironmentEnum;
use Mainto\RpcServer\RpcServer\RpcGatewayUrlEnum;
use Mainto\RpcServer\RpcUtil\RpcVarUpload;
use Mainto\RpcServer\Util\OS;

if (!function_exists('get_classes')) {
    function get_classes ($dir): array {
        $files = get_files($dir);
        $classes = [];
        foreach ($files as $file) {
            if (Str::endsWith($file, '.php')) {
                $class = get_class_from_file($file);
                if ($class !== null) {
                    $classes[] = $class;
                }
            }
        }

        return $classes;
    }
}

if (!function_exists('get_class_from_file')) {
    function get_class_from_file ($phpFile): ?string {
        $content = file_get_contents($phpFile);
        $tokens = token_get_all($content);
        $namespace = '';
        for ($index = 0; isset($tokens[$index]); $index++) {
            if (!isset($tokens[$index][0])) {
                continue;
            }
            if (T_NAMESPACE === $tokens[$index][0]) {
                $index += 2; // Skip namespace keyword and whitespace
                while (isset($tokens[$index]) && is_array($tokens[$index])) {
                    $namespace .= $tokens[$index++][1];
                }
            }
            if (T_CLASS === $tokens[$index][0] && T_WHITESPACE === $tokens[$index + 1][0] && T_STRING === $tokens[$index + 2][0]) {
                $index += 2; // Skip class keyword and whitespace
                return $namespace.'\\'.$tokens[$index][1];

                # break if you have one class per file (psr-4 compliant)
                # otherwise you'll need to handle class constants (Foo::class)
            }
        }

        return null;
    }
}

if (!function_exists('safe_fork')) {
    function safe_fork (): int {
        $pid = pcntl_fork();
        if ($pid == 0) {
            disconnect_all();
        }

        return $pid;
    }
}

if (!function_exists('disconnect_all')) {
    function disconnect_all () {
        RpcClientFactory::clearAllClient();

        $dbConnections = DB::getConnections();
        foreach ($dbConnections as $name => $connection) {
            Log::warning("disconnect $name db by disconnect_all");
            DB::purge($name);
        }

        $redisConnections = \Illuminate\Support\Facades\Redis::connections();
        if ($redisConnections) {
            foreach ($redisConnections as $name => $connection) {
                $connection->disconnect();
                \Illuminate\Support\Facades\Redis::purge($name);
                Log::warning("disconnect $name redis by disconnect_all");
            }
        }

        $allResource = get_resources();
        foreach ($allResource as $resource) {
            switch (get_resource_type($resource)) {
                case "Socket":
                    socket_close($resource);
                    break;
                case "stream":
                    $metadata = stream_get_meta_data($resource);
                    if (intval($resource) <= 3 || $metadata['stream_type'] == 'STDIO') {
                        break;
                    } else {
                        fclose($resource);
                    }
                    break;
                case "stream-context":
                    // 不需要处理
                    break;
                default:
                    Log::warning("unknown resource type :".get_resource_type($resource));
            }
        }
    }

    if (OS::getOs() == 'linux') {
        $cmd = "ls -al /proc/".getmypid()."/fd/";
        if (Str::contains(`$cmd`, 'socket')) {
            if (extension_loaded('xdebug') && xdebug_code_coverage_started()) {
                throw new RuntimeException("disconnect_all fail, has not disconnected socket");
            }
        }
    }
}

if (!function_exists('must_get_namespace')) {
    function must_get_namespace ($file) {
        $namespace = null;

        if (preg_match('#^namespace\s+(.+?);$#sm', file_get_contents($file), $m)) {
            $namespace = $m[1];
        }
        if (!$namespace) {
            throw new RuntimeException(sprintf("get namespace fail from %s", $file));
        }

        return $namespace;
    }
}

if (!function_exists('get_doc')) {
    function get_doc (string $fullDoc) {
        $doc = '';
        foreach (explode("\n", $fullDoc) as $line) {
            if (strpos($line, '@') === false && strpos($line, '/') === false) {
                $_doc = trim(str_replace('*', '', $line));
                if (!$_doc) continue;
                $doc .= $_doc."\n";
            }
        }

        return trim($doc);
    }
}

if (!function_exists('body_encode')) {
    function body_encode ($body): string {
        if ($body instanceof Jsonable) {
            $encode = (string)$body->toJson();
        } else {
            $encode = json_encode($body instanceof Arrayable ? $body->toArray() : $body);

            if ($encode === false) {
                throw new RpcRuntimeException("the response can not convert to json: ".json_last_error_msg());
            }
        }

        return $encode;
    }
}


if (!function_exists('rpc_gateway_url')) {
    /**
     * @param $environment
     * @return bool|mixed|string
     */
    function rpc_gateway_url ($environment) {
        switch ($environment) {
            case RpcEnvironmentEnum::Stable:
                return RpcGatewayUrlEnum::Stable;
            case RpcEnvironmentEnum::Dev:
                return RpcGatewayUrlEnum::Dev;
            case RpcEnvironmentEnum::Pre:
                return RpcGatewayUrlEnum::Pre;
            case RpcEnvironmentEnum::Release:
                return RpcGatewayUrlEnum::Release;
            case RpcEnvironmentEnum::ReleaseDeploy:
                if (env('RPC_GATEWAY_URL')) {
                    return env('RPC_GATEWAY_URL');
                }
        }

        return RpcGatewayUrlEnum::Local;
    }
}


if (!function_exists('short_package_name')) {
    /**
     * @param $name
     * @return string
     */
    function short_package_name ($name): string {
        $name = preg_replace('/.*[aA]pp\\\\[\w]+\\\\/', '', $name);

        return preg_replace('/\\\\?Controllers\\\\/', '\\', $name);
    }
}

if (!function_exists('rpc_environment')) {
    /**
     * @param $hostname
     * @return bool|mixed|string
     */
    function rpc_environment ($hostname) {
        $hostnameArr = explode('-', $hostname);
        if (count($hostnameArr) >= 3) {
            $tag = $hostnameArr[count($hostnameArr) - 3];
            switch ($tag) {
                case RpcEnvironmentEnum::Stable:
                case RpcEnvironmentEnum::Pre:
                case RpcEnvironmentEnum::Dev:
                    return $tag;
                case RpcEnvironmentEnum::Release:
                    if (env('RPC_GATEWAY_URL')) {
                        return RpcEnvironmentEnum::ReleaseDeploy;
                    }
                    return RpcEnvironmentEnum::Release;
            }
        }

        return RpcEnvironmentEnum::Local;
    }
}

if (!function_exists('path_join')) {
    function path_join (...$args) {
        $path = "";
        foreach ($args as $arg) {
            if (!$arg) continue;

            $path .= $path ? (DIRECTORY_SEPARATOR.trim($arg, DIRECTORY_SEPARATOR)) : rtrim($arg, DIRECTORY_SEPARATOR);
        }

        return $path;
    }
}

if (!function_exists('remove_all')) {
    function remove_all ($dir) {
        $files = array_diff(scandir($dir), ['.', '..']);
        foreach ($files as $file) {
            (is_dir("$dir/$file")) ? remove_all("$dir/$file") : unlink("$dir/$file");
        }
        return rmdir($dir);
    }
}

if (!function_exists('rpc_petty_exception_trace')) {
    function rpc_petty_exception_trace (Throwable $throwable, bool $upload = false): array {
        $traceArray = [];
        $firstNonFramework = -1;
        foreach ($throwable->getTrace() as $line) {
            $traceArray[] = $line;
            if ($firstNonFramework == -1
                && strpos($line['file'], 'php-rpc-framework/src') === false
                && strpos($line['file'], 'vendor/mainto/micro-bridge-') === false) {
                $firstNonFramework = count($traceArray) - 1;
            }

            if (strpos($line['file'], 'php-rpc-framework/src/RpcServer/RpcInvoke.php') !== false
                && (
                    strpos($line['function'], 'call_user_func_array') !== false
                    || strpos($line['function'], 'invokeThroughMiddleware') !== false
                )) {
                $traceArray[] = [
                    'file'     => '[framework trace]',
                    'line'     => 0,
                    'class'    => '',
                    'type'     => '',
                    'function' => '',
                    'args'     => [],
                ];
                break;
            }
        }

        if ($firstNonFramework == -1) $firstNonFramework = 0;

        $otherField = [];
        if ($upload) {
            $traceUrl = RpcVarUpload::getInstance()->upload($traceArray);
            if ($traceUrl) $otherField['error_trace_url'] = "https://cc.ink/00000d0sNL2g#$traceUrl";
        }

        return array_merge($otherField, [
            'error_trace' => join("\n", array_map_with_Key(
                function ($trace, $index) {
                    $args = join(', ', array_map(
                            function ($arg) {
                                switch ($type = gettype($arg)) {
                                    case "object":
                                        return "class ".get_class($arg);
                                    case "string":
                                        $strlen = strlen($arg);
                                        if ($strlen < 10) {
                                            return $arg;
                                        } else {
                                            return substr($arg, 0, 10)."...($strlen byte)";
                                        }
                                    case "array":
                                        $size = count($arg);
                                        return "array($size)";
                                    case "boolean":
                                    case "integer":
                                    case "double":
                                        return "$type($arg)";
                                    default:
                                        return $type;
                                }
                            },
                            $trace['args'])
                    );
                    return [
                        $index => "#$index {$trace['file']}({$trace['line']}): {$trace['class']}{$trace['type']}{$trace['function']}($args)",
                    ];
                },
                $traceArray
            )),
            "error_file"  => $traceArray[$firstNonFramework]['file'],
            "error_line"  => $traceArray[$firstNonFramework]['line'],
        ]);
    }
}


if (!function_exists('get_object_public_vars')) {
    function get_object_public_vars ($object): array {
        return get_object_vars($object);
    }
}

if (!function_exists('to_array')) {
    function to_array (array $var): array {
        return array_map(function ($value) {
            if ($value instanceof Arrayable) {
                return $value->toArray();
            } elseif (is_array($value)) {
                return to_array($value);
            } else {
                return $value;
            }
        }, $var);
    }
}

if (!function_exists('to_array_map')) {
    function to_array_map (array $var): array {
        return array_map_with_Key(function ($value, $key) {
            if ($value instanceof Arrayable) {
                return [$key => $value->toArray()];
            } elseif (is_array($value)) {
                return [$key => to_array_map($value)];
            } else {
                return [$key => $value];
            }
        }, $var);
    }
}

if (!function_exists('array_map_with_Key')) {
    function array_map_with_Key (callable $callback, array $var): array {
        $map = [];
        foreach ($var as $key => $value) {
            $map += $callback($value, $key);
        }

        return $map;
    }
}

if (!function_exists('get_files')) {
    /**
     * Get files
     * @param $dir
     * @return array
     */
    function get_files ($dir) {
        $files = [];
        $scan = scandir($dir);
        foreach ($scan as $item) {
            if ($item == '.' || $item == '..') continue;
            if (is_dir($dir.'/'.$item)) {
                $files = array_merge($files, get_files($dir.'/'.$item));
            } else {
                $files[] = $dir.'/'.$item;
            }
        }
        return $files;
    }
}

if (!function_exists('get_git_branch_version')) {
    /**
     * Format the relative path as an absolute path with the current working directory as the reference
     *
     * @param null|string $basePath
     * @return string
     */
    function get_git_branch_version (?string $basePath = ''): string {
        if (!$version = env('INNER_FRAMEWORK_VERSION', env('CI_COMMIT_REF_SLUG', ''))) {

            $git_file = $basePath ? $basePath.'/.git/HEAD' : base_path('/.git/HEAD');

            if (!file_exists($git_file)) {
                throw new \RuntimeException('get version fail');
            }

            $version = trim(str_replace(['ref: refs/heads/'], '', file_get_contents($git_file)));
            if (!$version) throw new \RuntimeException('get version fail');
        }
        return $version;
    }
}

if (!function_exists('get_service_version')) {
    /**
     * 获取当前分支commit sha 版本
     *
     * @param string|null $basePath
     * @param bool $short
     * @return string
     */
    function get_service_version (?string $basePath = '', bool $short = true): string {
        if (!$version = env('SERVICE_VERSION')) {
            // 获取当前分支，env('CI_COMMIT_SHA')
            $sha = env('CI_COMMIT_SHA');
            if (empty($sha)) {
                $branchVersion = get_git_branch_version($basePath);
                $headsPath = '/.git/refs/heads/'.$branchVersion;
                $shaFile = $basePath ? $basePath.$headsPath : base_path($headsPath);
                $sha = file_exists($shaFile) ? file_get_contents($shaFile) : $branchVersion;
                if (!$sha) throw new \RuntimeException('get sha version fail');
            }
            // 判断sha是否需要缩写
            $version = $short ? substr($sha, 0, 8) : $sha;
        }
        return $version;
    }
}

if (!function_exists('copy_dir')) {
    function copy_dir ($src, $dst) {
        $dir = opendir($src);
        @mkdir($dst);
        while (false !== ($file = readdir($dir))) {
            if (($file != '.') && ($file != '..')) {
                if (is_dir($src.'/'.$file)) {
                    copy_dir($src.'/'.$file, $dst.'/'.$file);
                } else {
                    copy($src.'/'.$file, $dst.'/'.$file);
                }
            }
        }
        closedir($dir);
    }
}