<?php

namespace Mainto\RpcServer\RpcServer;

use Doctrine\Common\Annotations\AnnotationReader;
use Illuminate\Support\Str;
use Mainto\RpcServer\Exceptions\RpcRuntimeException;
use Mainto\RpcServer\RpcAnnotations\Alias;
use Mainto\RpcServer\RpcAnnotations\RpcApi;
use Mainto\RpcServer\RpcAnnotations\RpcAuthority;
use Mainto\RpcServer\RpcAnnotations\RpcCron;
use Mainto\RpcServer\RpcAnnotations\RpcDeprecated;
use Mainto\RpcServer\RpcAnnotations\RpcDisableTypeValidation;
use Mainto\RpcServer\RpcAnnotations\RpcErrorCode;
use Mainto\RpcServer\RpcAnnotations\RpcFormat;
use Mainto\RpcServer\RpcAnnotations\RpcHeader;
use Mainto\RpcServer\RpcAnnotations\RpcMessageHook;
use Mainto\RpcServer\RpcAnnotations\RpcMiddleware;
use Mainto\RpcServer\RpcAnnotations\RpcParam;
use Mainto\RpcServer\RpcAnnotations\RpcResponseHeader;
use Mainto\RpcServer\RpcAnnotations\RpcSupportEnv;
use Mainto\RpcServer\RpcAnnotations\RpcWebsocket;
use Mainto\RpcServer\RpcServer\Definition\Controller;
use Mainto\RpcServer\RpcServer\Definition\Method;
use Mainto\RpcServer\RpcServer\Middleware\Method\ArrayMapCheckMiddleware;
use Mainto\RpcServer\RpcServer\Middleware\Method\DataFillAndTypeCheckMiddleware;
use Mainto\RpcServer\RpcServer\Middleware\Method\RecoverMiddleware;
use Mainto\RpcServer\RpcServer\Middleware\Method\SessionMiddleware;
use Mainto\RpcServer\RpcServer\Middleware\Method\TooManyParametersCheckMiddleware;
use Mainto\RpcServer\RpcServer\Middleware\Method\ValidateMiddleware;
use Mainto\RpcServer\RpcServer\Middleware\MethodMiddleware;
use Mainto\RpcServer\RpcServer\Middleware\MethodMiddlewareApply;
use Mainto\RpcServer\Service\Struct\Config;
use ReflectionClass;
use ReflectionException;
use ReflectionMethod;
use RuntimeException;

class RpcRouter {

    /**
     * @var RpcRouter|null
     */
    private static ?RpcRouter $instance = null;

    /**
     * @var string
     */
    private static string $appDir = "";

    /**
     * alias Map
     * @var array ['controller' => 'registerName']
     */
    private array $registerMap = [];

    /**
     * @var array
     */
    private array $_pathUrlMap = [];

    /**
     * RpcRouter constructor.
     */
    private function __construct () {
    }

    public static function getInstance (): RpcRouter {
        if (self::$instance == null) {
            $instance = new RpcRouter();
            self::$instance = $instance;

            app('events')->listen('Illuminate\Console\Events\ArtisanStarting', function ($data) {
                // unless property
                self::$instance->_pathUrlMap = [];
            });
        }

        return self::$instance;
    }

    public static function clear () {
        self::$instance = null;
    }

    /**
     * @param string $appDir
     */
    public static function setAppDir (string $appDir): void {
        self::$appDir = $appDir;
    }

    /**
     * @return string
     */
    public static function getAppDir (): string {
        return self::$appDir;
    }

    /**
     * 注册一个RPC路由
     *
     * @param string|object $class
     * @param string|null $registerName
     */
    public function register ($class, string $registerName = null) {
        try {
            $refClass = new ReflectionClass($class);
            if (!$registerName) {
                $currentName = $refClass->getName();

                /**
                 * warp base name
                 * [App\a\b\c\Controllers\d\e\abcController] -> b\c\d\e\abcController
                 * [App\a\Controllers\d\eabcController] -> d\e\abcController
                 */

                $pregWarp = function ($preg, $replace, $currentName) use ($class) {
                    $registerName = preg_replace($preg, $replace, $currentName);
                    if ($currentName == $registerName) {
                        throw new RuntimeException($class.' not has alias and path error');
                    }

                    return $registerName;
                };
                $currentName = $pregWarp('/.*[aA]pp\\\\[\w]+\\\\/', '', $currentName);
                $currentName = $pregWarp('/\\\\?Controllers\\\\/', '\\', $currentName);
                $registerName = ltrim($currentName, '\\');
            }

            if (!Str::endsWith($registerName, 'Controller')) {
                throw new RuntimeException("class name must be use Controller end");
            }

            $registerName = substr($registerName, 0, -10);

            $this->addController($refClass, $registerName);
            event('rpc.framework.router.added', [
                'class'        => $class,
                'registerName' => $registerName,
            ]);
        } catch (ReflectionException $e) {
            /** @var RuntimeException $e */
            throw $e;
        }
    }

    /**
     * @param ReflectionClass $refClass
     * @param string $registerName
     * @throws ReflectionException
     */
    public function addController (ReflectionClass $refClass, string $registerName) {
        $definition = RpcDefinition::getInstance();

        if (isset($definition->controllers[$registerName])) {
            return;
        }

        $definition->controllers[$registerName] = $controller = new Controller($refClass->getName());
        $controller->registerName = $registerName;
        $controller->name = class_basename($registerName);

        $controllerEnableEnv = [];
        $annotations = app(AnnotationReader::class)->getClassAnnotations($refClass);
        $controllerMiddlewares = [];
        foreach ($annotations as $annotation) {
            if ($annotation instanceof Alias) {
                $controller->alias = $annotation->name;
            } elseif ($annotation instanceof RpcSupportEnv) {
                if ($annotation->enable && $annotation->disable) {
                    throw new RpcRuntimeException("please use one of enable and disable");
                }

                if ($annotation->enable) {
                    $controllerEnableEnv = $annotation->enable;
                }

                if ($annotation->disable) {
                    $controllerEnableEnv = array_diff(RpcEnvironmentEnum::All, $annotation->disable);
                }
            } elseif ($annotation instanceof RpcMiddleware) {
                $controllerMiddlewares[] = [
                    'class' => new $annotation->class,
                    'args'  => $annotation->args,
                ];
            }
        }

        $methods = $refClass->getMethods(ReflectionMethod::IS_PUBLIC);

        foreach ($methods as $method) {
            if (!$method->isStatic()) {
                $registerMethod = $this->addMethod($controller, $method, $controllerMiddlewares);
                if ($controllerEnableEnv && !$registerMethod->enableEnv) {
                    $registerMethod->enableEnv = $controllerEnableEnv;
                }
            }
        }

        $this->registerMap[$refClass->getName()] = $registerName;
    }

    /**
     * @param Controller $controller
     * @param ReflectionMethod $refMethod
     * @param array $controllerMiddlewares
     * @return Method|void
     * @throws ReflectionException
     */
    public function addMethod (Controller $controller, ReflectionMethod $refMethod, array $controllerMiddlewares = []) {
        $methodName = $refMethod->getShortName();
        if ($methodName[0] == "_") {
            return;
        }

        $controller->methods[$methodName] = $method = new Method($controller->registerName, $refMethod);
        $method->name = $methodName;

        $method->resolveReturnType();

        $method->loadDocument($refMethod->getDocComment());

        $annotations = app(AnnotationReader::class)->getMethodAnnotations($refMethod);
        $rpcParamAnnotations = [];
        $userMiddlewares = $controllerMiddlewares;
        foreach ($annotations as $annotation) {
            if ($annotation instanceof RpcApi) {
                $method->loadFromRpcApi($annotation);
            } elseif ($annotation instanceof RpcHeader) {
                $method->loadFromRpcHeader($annotation);
            } elseif ($annotation instanceof RpcErrorCode) {
                $method->loadFromRpcErrorCode($annotation);
            } elseif ($annotation instanceof RpcFormat) {
                $method->loadFromRpcFormat($annotation);
            } elseif ($annotation instanceof RpcResponseHeader) {
                $method->loadFromRpcResponseHeader($annotation);
            } elseif ($annotation instanceof RpcAuthority) {
                $method->loadFromRpcAuthority($annotation);
            } elseif ($annotation instanceof RpcMessageHook) {
                $method->loadFromRpcMessageHook($annotation);
            } elseif ($annotation instanceof RpcWebsocket) {
                $method->loadFromRpcWebsocket($annotation);
            } elseif ($annotation instanceof RpcDisableTypeValidation) {
                $method->loadFromRpcDisableTypeValidation($annotation);
            } elseif ($annotation instanceof RpcCron) {
                $method->loadFromRpcCron($annotation, $controller->registerName);
            } elseif ($annotation instanceof RpcParam) {
                $rpcParamAnnotations[$annotation->name] = $annotation;
                // $method->loadFromRpcParam($annotation);
            } elseif ($annotation instanceof Alias) {
                $method->loadFromAlias($annotation);
            } elseif ($annotation instanceof RpcSupportEnv) {
                $method->loadFromRpcSupportEnv($annotation);
            } elseif ($annotation instanceof RpcDeprecated) {
                $method->loadFromRpcDeprecated($annotation);
            } elseif ($annotation instanceof RpcMiddleware) {
                if (!is_subclass_of($annotation->class, MethodMiddleware::class)) {
                    throw new RpcRuntimeException($annotation->class.' class must be implement MethodMiddleware');
                }
                $userMiddlewares[] = [
                    'class' => app($annotation->class),
                    'args'  => $annotation->args,
                ];
            }
        }
        // 入参顺序以为准
        foreach ($method->getRefInstance()->getParameters() as $parameterRef) {
            if (isset($rpcParamAnnotations[$parameterRef->getName()])) {
                $method->loadFromRpcParam($rpcParamAnnotations[$parameterRef->getName()]);
                unset($rpcParamAnnotations[$parameterRef->getName()]);
            }
        }
        foreach ($rpcParamAnnotations as $rpcParamAnnotation) {
            $method->loadFromRpcParam($rpcParamAnnotation);
        }

        // check duplicate url
        if ($method->httpApi) {
            $urlKey = $method->httpApi->toString();
            if (isset($this->_pathUrlMap[$urlKey])) {
                throw new RuntimeException("already has this router: {$urlKey}");
            }

            $this->_pathUrlMap[$urlKey] = true;
        }

        $fullMethodName = $controller->registerName.'::'.$methodName;
        RpcInvoke::resetMethodMiddleware($fullMethodName);
        RpcInvoke::addMethodMiddleware($fullMethodName, DataFillAndTypeCheckMiddleware::getInstance());

        if ($method->checkValidation) {
            RpcInvoke::addMethodMiddleware($fullMethodName, ValidateMiddleware::getInstance());
        }

        if (config('app.env') != 'testing') {
            RpcInvoke::addMethodMiddleware($fullMethodName, RecoverMiddleware::getInstance());
        }
        RpcInvoke::addMethodMiddleware($fullMethodName, SessionMiddleware::getInstance());
        if (!in_array(config('rpc-server.environment'), [RpcEnvironmentEnum::Stable, RpcEnvironmentEnum::Pre])) {
            RpcInvoke::addMethodMiddleware($fullMethodName, ArrayMapCheckMiddleware::getInstance());
            RpcInvoke::addMethodMiddleware($fullMethodName, TooManyParametersCheckMiddleware::getInstance());
        }

        foreach ($userMiddlewares as $middlewareItem) {
            ['class' => $middleware, 'args' => $args] = $middlewareItem;
            if ($middleware instanceof MethodMiddlewareApply) {
                $middleware->apply($method);
            }
            RpcInvoke::addMethodMiddleware($fullMethodName, $middleware, $args);
        }

        // validate and cache
        $_ = $method->getParameters();

        return $method;
    }

    public function getController (string $registerClassName): ?Controller {
        return RpcDefinition::getInstance()->controllers->get($registerClassName);
    }

    public function getControllerMethod (string $registerClassName, $methodName): ?Method {
        return RpcDefinition::getInstance()->controllers->get($registerClassName)->methods->get($methodName);
    }

    public function getConfig (): Config {
        return Config::createFromDefinition(RpcDefinition::getInstance());
    }

    /**
     * 获取控制器注册名
     *
     * @param $class
     * @return string|null
     */
    public function getControllerRegisterName ($class): ?string {
        return $this->registerMap[$class] ?? null;
    }

    /**
     * 获取基础配置信息
     *
     * @return array
     */
    public function getBaseRouterConfig (): array {
        $struct = Config::createFromDefinition(RpcDefinition::getInstance());

        $config = [
            'name'       => config('rpc-server.service_name'),
            'paths'      => $struct->getPaths()->toArray(),
            'crons'      => [],
            'messages'   => [],
            'websockets' => $struct->getWebsockets()->toArray(),
        ];

        if (config('rpc-server.cron.enable')) {
            $config['crons'] = $struct->getCrons()->toArray();
        }

        if (config('rpc-server.sidecar.rabbitmq.enable')) {
            $config['messages'] = $struct->getMessages()->toArray();
        }

        if (config('rpc-server.sidecar.rocketmq.enable')) {
            $config['message_queues'] = $struct->getMessageQueues()->toArray();
        }

        return $config;
    }
}