<?php

namespace Mainto\RpcServer\Tracing\Support\OTEL;

use Mainto\RpcServer\Tracing\Source;
use Mainto\RpcServer\Util\Types\Map;
use OpenTelemetry\API\Trace as API;
use OpenTelemetry\Contrib\Otlp\AttributesConverter;
use OpenTelemetry\Contrib\Otlp\ProtobufSerializer;
use Opentelemetry\Proto\Collector\Trace\V1\ExportTraceServiceRequest;
use Opentelemetry\Proto\Common\V1\InstrumentationScope;
use Opentelemetry\Proto\Common\V1\KeyValue;
use Opentelemetry\Proto\Resource\V1\Resource as Resource_;
use Opentelemetry\Proto\Trace\V1\ResourceSpans;
use Opentelemetry\Proto\Trace\V1\ScopeSpans;
use Opentelemetry\Proto\Trace\V1\Span;
use Opentelemetry\Proto\Trace\V1\Span\Event;
use Opentelemetry\Proto\Trace\V1\Span\Link;
use Opentelemetry\Proto\Trace\V1\Span\SpanKind;
use Opentelemetry\Proto\Trace\V1\Status;
use Opentelemetry\Proto\Trace\V1\Status\StatusCode;
use OpenTelemetry\SDK\Common\Attribute\AttributesInterface;
use OpenTelemetry\SDK\Common\Instrumentation\InstrumentationScopeInterface;
use OpenTelemetry\SDK\Resource\ResourceInfo;
use OpenTelemetry\SDK\Trace\SpanDataInterface;

class Converter {


    public function __construct(?ProtobufSerializer $serializer = null)
    {
        $this->serializer = $serializer ?? ProtobufSerializer::getDefault();
    }

    public function convert(iterable $spans, Source $source): ExportTraceServiceRequest
    {
        $pExportTraceServiceRequest = new ExportTraceServiceRequest();

        foreach ($spans as $span) {
            $pExportTraceServiceRequest->getResourceSpans()[]
                = $this->convertSpan($span, $source);
        }

        return $pExportTraceServiceRequest;
    }

    /**
     * @param \Mainto\RpcServer\Tracing\Span $span
     * @param Source $source
     * @return array
     */
    public function convertSpan (Span $span, Source $source): array {
        $row = [
            'host'          => $source->getHost(),
            'service'       => $source->getServiceName(),
            'otlp.name'     => $source->getSdkVersion()->getName(),
            'otlp.version'  => $source->getSdkVersion()->getVersion(),
            'name'          => $span->getName(),
            'kind'          => \Mainto\RpcServer\Tracing\Interfaces\SpanKind::TYPES[$span->getKind()] ?? 'server',
            'traceID'       => $span->getTraceId(),
            'spanID'        => $span->getSpanId(),
            'parentSpanID'  => $span->getParentId(),
            'start'         => $span->getStart(),
            'end'           => $span->getEnd(),
            'duration'      => $span->getDuration(),
            'statusCode'    => $span->getStatusCode(),
            'statusMessage' => $span->getStatusMessage(),
            'resource'      => $source->getResources(),
        ];
        // 合并数组
        $row['attribute'] = $span->getAttributes()->merge($source->getAttributes());
        // 判断是否为空
        if (!empty($span->getEvents())) {
            // 初始化
            $row['logs'] = [];
            // 遍历循环
            foreach ($span->getEvents() as $event) {
                // 追加数据
                $row['logs'][] = [
                    'name'       => $event->getName(),
                    'timestamp'  => $event->getTimestamp(),
                    'attributes' => $event->getAttributes(),
                ];
            }
        }
        // 返回
        return $row;
    }

}
