<?php

namespace Mainto\RpcServer\Tracing\Extend;

use Mainto\RpcServer\Exceptions\RpcRuntimeException;
use Mainto\RpcServer\RpcUtil\JsonHelper;
use Mainto\RpcServer\Tracing\Interfaces\SpanKind;
use Mainto\RpcServer\Tracing\Span;
use Throwable;

/**
 * 链路AOP切面
 */
abstract class TraceAOP {
    /**
     * @var string
     */
    protected string $className = '';

    /**
     * @var object|null
     */
    protected ?object $classObject = null;

    /**
     * @var string
     */
    protected string $spanName = '';

    /**
     * @var string
     */
    protected string $spanAttributeKey = 'args';

    /**
     * @var int
     */
    protected int $spanKind = SpanKind::KIND_INTERNAL;

    /**
     * @return mixed
     */
    protected function getClass () {
        if (!is_null($this->classObject)) {
            return $this->classObject;
        }
        // 判断是否设置 className属性
        if (empty($this->className)) {
            throw new RpcRuntimeException(sprintf(
                '%s ClassObject And ClassName Is Empty', get_class($this)
            ));
        }
        // 判断类是否存在
        if (!class_exists($this->className)) {
            throw new RpcRuntimeException(sprintf('Class Not Found %s', $this->className));
        }
        // 实例化、赋值、并返回
        return $this->classObject = new $this->className();
    }

    /**
     * @param string $method
     * @param array $args
     * @return string
     */
    protected function getSpanName (string $method = '', array $args = []): string {
        if (empty($this->spanName)) {
            return sprintf('TraceAOP %s::%s', get_class($this->getClass()), $method);
        }
        return sprintf($this->spanName, $method);
    }

    /**
     * @return string
     */
    protected function getAttributeKey (): string {
        return $this->spanAttributeKey;
    }

    /**
     * @param $method
     * @param $args
     * @return mixed
     * @throws Throwable
     */
    public function __call ($method, $args) {
        $object = $this->getClass();
        if (!method_exists($object, $method)) {
            throw new RpcRuntimeException(sprintf(
                'Method %s Not Exist In %s', $method, get_class($object)
            ));
        }
        // 进入链路处理
        $span = Span::start($this->getSpanName($method, $args), null, $this->spanKind);
        try {
            if (!empty($args)) {
                $span->setAttribute($this->getAttributeKey(), JsonHelper::encodeWithUnescapedAndError($args));
            }
            return call_user_func_array([$object, $method], $args);
        } catch (Throwable $e) {
            $span->statusError($e->getMessage());
            throw $e;
        } finally {
            $span->end();
        }
    }
}