<?php

namespace Mainto\RpcServer\Tracing;

use Mainto\RpcServer\Tracing\Interfaces\SpanKind;
use Mainto\RpcServer\Tracing\Traits\Attributes;
use Mainto\RpcServer\Tracing\Traits\Links;
use Mainto\RpcServer\Tracing\Traits\Events;
use Mainto\RpcServer\Tracing\Traits\Resource;
use Mainto\RpcServer\Tracing\Traits\Sampled;
use Mainto\RpcServer\Tracing\Traits\Status;
use Mainto\RpcServer\Util\Types\Map;
use OpenTelemetry\SDK\Common\Time\ClockFactory;
use OpenTelemetry\SDK\Trace\StatusDataInterface;

class Span {
    use Attributes, Links, Events, Resource, Status, Sampled;

    /**
     * @var string
     */
    protected string $name;

    /**
     * @var int
     */
    protected int $kind;

    /**
     * @var SpanContext
     */
    protected SpanContext $context;

    /**
     * @var SpanContext
     */
    protected SpanContext $parentContext;

    /**
     * @var Span|null
     */
    protected ?Span $parent = null;

    /**
     * @var Tracer
     */
    protected Tracer $tracer;

    private int $totalRecordedEvents = 0;
    private StatusDataInterface $status;
    private int $startEpochNanos = 0;
    private int $endEpochNanos = 0;
    private bool $hasEnded = false;

    /**
     * @param string $name
     * @param SpanContext|null $parentContext
     * @param int $kind
     */
    private function __construct (string $name = '', SpanContext $parentContext = null, int $kind = SpanKind::KIND_INTERNAL) {
        $this->tracer = Tracer::getInstance();

        $this->startEpochNanos = ClockFactory::getDefault()->now();
        $this->name = $name;
        $this->kind = $kind;
        $this->statusOk();
        $this->attributes = new Map();
        $this->resource = new Map();
        // 获取当前spanContext中的span
        $this->parent = $parentContext->getSpan();
        $this->parentContext = $parentContext;

        $this->context = new SpanContext(
            $this->tracer->getTraceId(),
            $this->tracer->getIdGenerator()->generateSpanId(),
            $parentContext->getSpanId(),
            $this,
        );

        $this->tracer->getProcessor()->onStart($this);
        $this->setSampled($this->tracer->isSampled());
    }

    /**
     * 快速调用
     *
     * @param string $name
     * @param SpanContext|null $context
     * @param int $kind
     *
     * @return Span
     */
    public static function start (string $name, SpanContext $context = null, int $kind = SpanKind::KIND_INTERNAL): Span {
        if (is_null($context)) {
            $context = Context::getSpanContext();
        }
        if (Tracer::getInstance()->isSampled() === false) {
            return NoopSpan::getInstance($name, $context, $kind);
        }
        return new Span($name, $context, $kind);
    }

    /**
     * @return string
     */
    public function getName (): string {
        return $this->name;
    }

    /**
     * @param string $name
     * @return $this
     */
    public function setName (string $name): self {
        if ($this->hasEnded) {
            return $this;
        }
        $this->name = $name;

        return $this;
    }

    /**
     * @return int
     */
    public function getKind (): int {
        return $this->kind;
    }

    /**
     * @return string
     */
    public function getSpanId (): string {
        return $this->context->getSpanId();
    }

    /**
     * @return string
     */
    public function getTraceId (): string {
        return $this->context->getTraceId();
    }

    /**
     * @return string
     */
    public function getParentId (): string {
        return $this->context->getParentSpanId();
    }

    /**
     * @return SpanContext
     */
    public function getContext (): SpanContext {
        return $this->context;
    }

    /**
     * @return Span|null
     */
    public function parent (): ?Span {
        return $this->parent;
    }

    public function getParentContext(): SpanContext
    {
        return $this->parentContext;
    }

    /**
     * @param int $kind
     * @return $this
     */
    public function setKind (int $kind): self {
        $this->kind = $kind;
        return $this;
    }

    public function getStartEpochNanos(): int
    {
        return $this->startEpochNanos;
    }

    public function getEndEpochNanos(): int
    {
        return $this->endEpochNanos;
    }

    public function isRecording(): bool
    {
        return !$this->hasEnded;
    }

    public function hasEnded(): bool
    {
        return $this->hasEnded;
    }

    public function getDuration(): int
    {
        return ($this->hasEnded ? $this->endEpochNanos : ClockFactory::getDefault()->now()) - $this->startEpochNanos;
    }

    /**
     * span结束
     */
    public function end (): void {
        if ($this->hasEnded) {
            return;
        }

        $this->hasEnded = true;

        $this->endEpochNanos = ClockFactory::getDefault()->now();

        $this->tracer->getProcessor()->onEnd($this);
    }
}
