<?php

namespace Mainto\RpcServer\RpcUtil\Tool\LogDriver;

use Mainto\RpcServer\Protocol\Request\Request;
use Mainto\RpcServer\Tracing\SpanArray;
use Throwable;

class Local extends LogDriveAbstract {

    public function __construct (?string $level) {
        parent::__construct($level);
    }

    /**
     * send messages to local file
     *
     * @param string $source
     * @param string $logStore
     * @param $contents
     */
    private function send (string $source, string $logStore, $contents) {
        $content = sprintf("[%s] %s: %s\n", date('Y-m-d H:i:s'), $source, $this->formatMessage($contents));
        @file_put_contents(storage_path("logs/rpc_log_{$logStore}.log"), $content, FILE_APPEND);
    }

    public function logThrowable ($request, Throwable $e, string $source = "") {
        if ($source == "") {
            $traceInfo = $this->getTraceInfo(-1);
            $source = $traceInfo['source'];
        }

        $this->send($source, "error", [
            "request"       => $this->formatMessage($request),
            "error_name"    => class_basename($e),
            "error_code"    => $e->getCode(),
            "error_content" => $e->getMessage(),
            "error_file"    => $e->getFile(),
            "error_line"    => $e->getLine(),
            "error_trace"   => rpc_exception_trace($e),
        ]);
    }

    /**
     * log throwable error log
     *
     * @param Request $request
     * @param Throwable $e
     * @param string $source
     */
    public function logRequestThrowable (Request $request, Throwable $e, string $source = "") {
        if ($source == "") {
            $traceInfo = $this->getTraceInfo(-1);
            $source = $traceInfo['source'];
        }

        $this->send($source, "error", [
            "header"        => $request->getAllHeaders(),
            "request"       => $request->getAllParams(),
            "error_name"    => class_basename($e),
            "error_code"    => $e->getCode(),
            "error_content" => $e->getMessage(),
            "error_file"    => $e->getFile(),
            "error_line"    => $e->getLine(),
            "error_trace"   => rpc_exception_trace($e),
        ]);
    }

    /**
     * log some messgae
     *
     * @param string $level
     * @param string $title
     * @param $message
     */
    protected function log (string $level, string $title, $message) {
        if (!$this->isHandling($level)) {
            return;
        }
        $traceInfo = $this->getTraceInfo();
        $this->send($traceInfo['source'], "log", [
            "offset"  => $traceInfo['file'],
            "level"   => $level,
            "title"   => $title,
            "content" => $message,
        ]);
    }

    /**
     * log debug message
     *
     * @param string $title
     * @param $message
     */
    public function debug (string $title, $message) {
        $this->log("DEBUG", $title, $message);
    }

    /**
     * log info message
     *
     * @param string $title
     * @param $message
     */
    public function info (string $title, $message) {
        $this->log("INFO", $title, $message);
    }

    /**
     * log notice message
     *
     * @param string $title
     * @param $message
     */
    public function notice (string $title, $message) {
        $this->log("NOTICE", $title, $message);
    }

    /**
     * log warning message
     *
     * @param string $title
     * @param $message
     */
    public function warning (string $title, $message) {
        $this->log("WARNING", $title, $message);
    }

    /**
     * log error message
     *
     * @param string $title
     * @param $message
     */
    public function error (string $title, $message) {
        $this->log("ERROR", $title, $message);
    }

    /**
     * log critical message
     *
     * @param string $title
     * @param $message
     */
    public function critical (string $title, $message) {
        $this->log("CRITICAL", $title, $message);
    }

    /**
     * @param SpanArray $spans
     */
    public function tracer (SpanArray $spans) {
        $this->send("TRACER", 'tracer', $spans->toArray());
    }
}