<?php

namespace Mainto\RpcServer\RpcUtil\Tool\LogDriver;

use Illuminate\Support\Facades\Log;
use Mainto\RpcServer\Exceptions\RpcRuntimeException;
use Mainto\RpcServer\Protocol\Common\Body;
use Mainto\RpcServer\Protocol\Common\Types\RequestHeaderType;
use Mainto\RpcServer\Protocol\Common\Types\ResponseHeaderType;
use Mainto\RpcServer\Protocol\Request\Extend\RequestExtendHeader;
use Mainto\RpcServer\Protocol\Request\Request;
use Mainto\RpcServer\Protocol\Response\Extend\ResponseExtendError;
use Mainto\RpcServer\RpcClient\RpcClient;
use Mainto\RpcServer\RpcClient\RpcClientFactory;
use Mainto\RpcServer\RpcClient\RpcClientWeakRef;
use Mainto\RpcServer\RpcServer\RpcInvoke;
use Mainto\RpcServer\Tracing\SpanArray;
use RuntimeException;
use Throwable;

class SidecarSLS extends LogDriveAbstract {
    private static ?Request $cacheRequest = null;

    /**
     * @var RpcClientWeakRef
     */
    private RpcClientWeakRef $clientRef;

    public function __construct () {
        if (self::$cacheRequest === null) {
            $requestExtendHeader = new RequestExtendHeader();
            self::$cacheRequest = new Request();
            self::$cacheRequest->setType(RequestHeaderType::SidecarSLSType);
            self::$cacheRequest->setExtend($requestExtendHeader);
        }

        $this->clientRef = new RpcClientWeakRef([RpcClientFactory::SlsClient]);
    }

    /**
     * @return RpcClient
     */
    protected function getClient (): RpcClient {
        return $this->clientRef->getClient();
    }

    public function clearClient () {
        $this->clientRef->clearClient();
    }

    /**
     * @param $request
     * @param Throwable $e
     * @param string $source
     * @return void
     */
    public function logThrowable ($request, Throwable $e, string $source = "") {
        if ($source == "") {
            $traceInfo = $this->getTraceInfo(-1);
            $source = $traceInfo['source'];
        }

        $this->send($source, "error", [
            "request"       => $this->formatMessage($request),
            "error_name"    => class_basename($e),
            "error_code"    => $e->getCode(),
            "error_content" => $e->getMessage(),
            "error_file"    => $e->getFile(),
            "error_line"    => $e->getLine(),
            "error_trace"   => rpc_exception_trace($e),
        ]);
    }

    /**
     * log throwable error log
     *
     * @param Request $request
     * @param Throwable $e
     * @param string $source
     */
    public function logRequestThrowable (Request $request, Throwable $e, string $source = "") {
        if ($source == "") {
            $traceInfo = $this->getTraceInfo(-1);
            $source = $traceInfo['source'];
        }

        $this->send($source, "error", [
            "header"        => $request->getAllHeaders(),
            "request"       => $request->getAllParams(),
            "error_name"    => class_basename($e),
            "error_code"    => $e->getCode(),
            "error_content" => $e->getMessage(),
            "error_file"    => $e->getFile(),
            "error_line"    => $e->getLine(),
            "error_trace"   => rpc_exception_trace($e),
        ]);
    }

    /**
     * send message to sidecar sls service
     *
     * @param string $source
     * @param string $logStore
     * @param array $contents
     */
    private function send (string $source, string $logStore, array $contents) {
        try {
            /** @var RequestExtendHeader $ext */
            $ext = self::$cacheRequest->getExt();
            $ext->setHeader("channel", [config('rpc-server.sidecar.sls.channel')[$logStore]]);

            $contents['method'] = $source;
            $contents['service_name'] = config('rpc-server.service_name');
            $contents['namespace'] = config('rpc-server.sidecar.namespace');
            self::$cacheRequest->setBody(Body::newJsonBody(json_encode($contents)));

            self::$cacheRequest->setTraceId(RpcInvoke::getCurrentTraceId());
            $response = $this->getClient()->Do(self::$cacheRequest);
            switch ($response->getType()) {
                case ResponseHeaderType::ReturnOKType:
                    break;
                case ResponseHeaderType::ReturnSystemErrType:
                case ResponseHeaderType::ReturnErrType:
                    $extend = $response->getExt();
                    if ($extend instanceof ResponseExtendError) {
                        throw new RpcRuntimeException($extend->getErrMsg(), $extend->getErrorCode());
                    }

                    throw new RuntimeException("unknown response ext");
                default:
                    throw new RuntimeException("unknown response");
            }
        } catch (Throwable $e) {
            if ($e instanceof RpcRuntimeException) {
                throw $e;
            }
            $this->clearClient();
            Log::error($e, $contents);
            throw new RuntimeException("can not find this sls server", 0, $e);
        }
    }

    /**
     * log info message
     *
     * @param string $title
     * @param $message
     */
    public function info (string $title, $message) {
        $this->log("INFO", $title, $message);
    }

    /**
     * log some message
     *
     * @param string $level
     * @param string $title
     * @param $message
     */
    protected function log (string $level, string $title, $message) {
        if (!$this->isHandling($level)) {
            return;
        }

        $traceInfo = $this->getTraceInfo();
        $this->send($traceInfo['source'], "log", [
            "offset"  => $traceInfo['file'],
            "level"   => $level,
            "title"   => $title,
            "content" => $this->formatMessage($message),
        ]);
    }

    /**
     * log debug message
     *
     * @param string $title
     * @param $message
     */
    public function debug (string $title, $message) {
        $this->log("DEBU", $title, $message);
    }

    /**
     * log notice message
     *
     * @param string $title
     * @param $message
     */
    public function notice (string $title, $message) {
        $this->log("NOTI", $title, $message);
    }

    /**
     * log warning message
     *
     * @param string $title
     * @param $message
     */
    public function warning (string $title, $message) {
        $this->log("WARN", $title, $message);
    }

    /**
     * log error message
     *
     * @param string $title
     * @param $message
     */
    public function error (string $title, $message) {
        $this->log("ERRO", $title, $message);
    }

    /**
     * log critical message
     *
     * @param string $title
     * @param $message
     */
    public function critical (string $title, $message) {
        $this->log("CRIT", $title, $message);
    }

    /**
     * @param SpanArray $spans
     */
    public function tracer (SpanArray $spans) {
        $this->send('TRACER', 'tracer', $spans->toArray());
    }
}
