<?php

namespace Mainto\RpcServer\Tracing;

use Mainto\RpcServer\Tracing\Interfaces\ExporterInterface;

class Processor {
    /**
     * @var Span[]
     */
    protected array $spans = [];

    /**
     * @var ExporterInterface
     */
    protected ExporterInterface $exporter;

    /**
     * @var Source
     */
    protected Source $source;

    /**
     * @param ExporterInterface $exporter
     * @param Source $source
     */
    public function __construct (ExporterInterface $exporter, Source $source) {
        $this->exporter = $exporter;
        $this->source = $source;
    }

    /**
     * span 开始
     *
     * @param Span $span
     */
    public function onStart (Span $span): void {
        $this->spans[$span->getSpanId()] = $span;
        Context::setSpanContext($span->getContext());
    }

    /**
     * span 结束
     *
     * @param Span $span
     */
    public function onEnd (Span $span): void {
        if (!empty($span->parent())) {
            Context::setSpanContext($span->parent()->getContext());
        }
        // 判断是否需要上报
        if ($span->isSampled()) {
            $this->exporter->export([$span], $this->source);
        }
        // 减少已上报数据的内存占用
        unset($this->spans[$span->getSpanId()]);
        // 防止内存泄漏
        $span->getContext()->clearSpan();
    }

    /**
     * 退出
     */
    public function shutdown (): void {
        if (!empty($this->spans)) {
            $this->forceFlush();
        }
    }

    /**
     * 强制提交
     */
    public function forceFlush (): void {
        // 遍历循环, 进行最终检查
        foreach ($this->spans as $span) {
            // 判断是否结束
            if ($span->isRecording()) {
                $span->end();
            }
        }
        $this->spans = [];
    }
}