<?php


namespace Mainto\RpcServer\RpcServer\Definition;


use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Support\Str;
use Mainto\RpcServer\Exceptions\RpcRuntimeException;
use Mainto\RpcServer\RpcAnnotations\Alias;
use Mainto\RpcServer\RpcAnnotations\RpcApi;
use Mainto\RpcServer\RpcAnnotations\RpcAuthority;
use Mainto\RpcServer\RpcAnnotations\RpcCron;
use Mainto\RpcServer\RpcAnnotations\RpcDeprecated;
use Mainto\RpcServer\RpcAnnotations\RpcDisableTypeValidation;
use Mainto\RpcServer\RpcAnnotations\RpcException;
use Mainto\RpcServer\RpcAnnotations\RpcFormat;
use Mainto\RpcServer\RpcAnnotations\RpcHeader;
use Mainto\RpcServer\RpcAnnotations\RpcMessageHook;
use Mainto\RpcServer\RpcAnnotations\RpcParam;
use Mainto\RpcServer\RpcAnnotations\RpcResponseHeader;
use Mainto\RpcServer\RpcAnnotations\RpcSupportEnv;
use Mainto\RpcServer\RpcAnnotations\RpcWebsocket;
use Mainto\RpcServer\RpcServer\RpcContext;
use Mainto\RpcServer\RpcServer\RpcDefinition;
use Mainto\RpcServer\RpcServer\RpcEnvironmentEnum;
use Mainto\RpcServer\RpcServer\RpcObject;
use Mainto\RpcServer\Util\Language;
use Mainto\RpcServer\Util\ObjectMapper\ArrayTypeParser;
use Mainto\RpcServer\Util\ObjectMapper\ClassRef;
use Mainto\RpcServer\Util\ObjectMapper\MapTypeParser;
use Mainto\RpcServer\Util\Types\Map;
use ReflectionException;
use ReflectionMethod;
use RuntimeException;

class Method implements Arrayable {
    /**
     * @var string
     */
    public string $name;

    /**
     * @var string
     */
    public string $controllerName;

    /**
     * @var string
     */
    public string $alias = "";

    /**
     * @var array
     */
    public array $enableEnv = [];

    /**
     * @var Deprecated|null
     */
    public ?Deprecated $deprecated = null;

    /**
     * @var string
     */
    public string $description = "";

    /**
     * @var HttpApi|null
     */
    public ?HttpApi $httpApi = null;

    /**
     * @var ReqHeader[]
     */
    public array $reqHeaders = [];

    /**
     * @var RpcException[]
     */
    public array $rpcExceptions = [];

    /**
     * @var string
     */
    public string $requestType = RpcDefinition::EmptyType;

    /**
     * @var string
     */
    public string $responseType = RpcDefinition::AnyType;

    /**
     * @var string
     */
    public string $resSerialize = "Json";

    /**
     * @var ResHeader[]
     */
    public array $resHeaders = [];

    /**
     * @var bool
     */
    public bool $checkValidation = true;

    /**
     * @var bool
     */
    public bool $checkType = true;

    /**
     * @var array|null
     */
    public array $needAuthorities = [];

    /**
     * @var string|null
     */
    public ?string $wsRoom = null;

    /**
     * @var bool|null
     */
    public bool $wsNeedConnectMessage = false;

    /**
     * @var bool|null
     */
    public bool $wsNeedDisconnectMessage = false;

    /**
     * @var string|null
     */
    public ?string $messageHookName = null;

    /**
     * @var int
     */
    public int $messageHookDedicatedPoolSize = 0;

    /**
     * @var string|null
     */
    public ?string $cronSpec = null;

    /**
     * @var string|null
     */
    public ?string $cronHookName = null;

    /**
     * @var string
     */
    public string $since = "1.0.0";
    /**
     * @var string|null
     */
    public ?string $messageQueueHookName = null;
    /**
     * @var int|null
     */
    public ?int $messageConcurrencyNum = null;

    /**
     * @var Map
     */
    public Map $requestExamples;

    private ?ReflectionMethod $refInstance;
    /**
     * @var array|null [name]
     */
    private ?array $requestDeclareMap = null;
    /**
     * @var array|null [name]
     */
    private ?array $requestDeclareNames = null;
    /**
     * @var array|null [name => Property]
     */
    private ?array $parameters = null;

    public function __construct ($controllerRegisterName = null, ReflectionMethod $refInstance = null) {
        $this->refInstance = $refInstance;
        $this->alias = $refInstance ? $refInstance->getShortName() : "";
        if ($controllerRegisterName) {
            $this->controllerName = $controllerRegisterName;
        }

        $this->requestExamples = new Map();
    }

    public function path (): string {
        return $this->controllerName.'::'.$this->name;
    }

    /**
     * @return Parameter[]
     * @throws ReflectionException
     */
    public function getParameters (): array {
        if ($this->parameters === null) {
            $this->parameters = [];

            foreach ($this->refInstance->getParameters() as $parameterRef) {
                $parameter = new Parameter();
                $parameter->name = $parameterRef->name;
                $parameter->nullable = $parameterRef->allowsNull();

                if (isset($this->getRequestDeclareMap()[$parameterRef->name]) && !($parameterRef->getType() && class_exists($parameterRef->getType()->getName()))) {
                    /** @var Property $property */
                    $property = $this->getRequestDeclareMap()[$parameterRef->name];
                    $property->nullable = $parameterRef->allowsNull();
                    if ($parameterRef->isDefaultValueAvailable()) {
                        if (!RpcDefinition::getInstance()->inStructCache($property->type)) {
                            $property->default = $parameterRef->getDefaultValue();
                        }
                        $property->defaultValueAvailable = true;
                    } else {
                        if (!$property->defaultValueAvailable) {
                            $property->require = true;
                        }
                    }

                    $parameter->type = $property->type;
                    $parameter->default = $property->default;
                    $parameter->defaultValueAvailable = $property->defaultValueAvailable;
                    $parameter->require = $property->require;

                    $this->parameters[$parameterRef->name] = $parameter;
                    continue;
                }

                if ($parameterRef->getType()) {
                    $parameter->type = $parameterRef->getType()->getName();
                    if ($parameterRef->isDefaultValueAvailable()) {
                        $parameter->default = $parameterRef->getDefaultValue();
                        $parameter->defaultValueAvailable = true;
                    } else {
                        $parameter->require = true;
                    }

                    $this->parameters[$parameterRef->name] = $parameter;

                    if ($parameterRef->getType()->getName() == RpcContext::class) {
                        continue;
                    }

                    if (is_subclass_of($parameterRef->getType()->getName(), RpcObject::class)) {
                        if ($this->requestType != RpcDefinition::EmptyType) {
                            if (!is_subclass_of($this->requestType, RpcObject::class)) {
                                throw new RuntimeException("There can only be one in RpcObject and RpcParam: ".$this->path());
                            }

                            throw new RuntimeException("RpcObject only support one".$this->path());
                        }

                        $struct = RpcDefinition::getInstance()->objectStruct($parameterRef->getType()->getName());
                        $this->resolveStructPropertyFromHeaders($struct);
                        $this->requestType = $parameterRef->getType()->getName();
                        continue;
                    } else {
                        throw new RuntimeException("the rpc object only support subclass of RpcObject in [{$this->path()}] -> {$parameterRef->getType()->getName()}");
                    }
                }

                throw new RuntimeException("the rpc param [{$parameterRef->name}] not defined in [{$this->path()}]");
            }
        }

        return $this->parameters;
    }

    /**
     * @return array
     */
    public function getRequestDeclareMap (): array {
        if ($this->requestDeclareMap === null) {
            $this->requestDeclareMap = [];
            foreach (RpcDefinition::getInstance()->getStruct($this->requestType)->properties as $property) {
                $this->requestDeclareMap[$property->name] = $property;
            }
        }

        return $this->requestDeclareMap;
    }

    public function resolveStructPropertyFromHeaders (Struct $struct, &$resolved = []) {
        $resolved[$struct->fullName()] = true;
        foreach ($struct->properties as $property) {
            if ($property->fromHeader) {
                $header = new ReqHeader();
                $header->name = $property->fromHeader;
                $this->reqHeaders[$header->name] = $header;
            }

            if (RpcDefinition::getInstance()->inStructCache($property->type) && !isset($resolved[RpcDefinition::getInstance()->getStruct($property->type)->fullName()])) {
                $this->resolveStructPropertyFromHeaders(RpcDefinition::getInstance()->getStruct($property->type), $resolved);
            }
        }
    }

    public function loadFromRpcApi (RpcApi $rpcApi) {
        $url = $rpcApi->url;
        if ($rpcApi->addPrefix && config('rpc-server.url.enable_prefix')) {
            $url = path_join(config('rpc-server.url.prefix'), $url);
        }

        $this->httpApi = new HttpApi();
        $this->httpApi->method = $rpcApi->method;
        $this->httpApi->url = $url;
    }

    public function loadFromRpcHeader (RpcHeader $rpcHeader) {
        $header = new ReqHeader();
        $header->name = $rpcHeader->name;
        $this->reqHeaders[$header->name] = $header;
    }

    public function loadFromRpcException (RpcException $rpcException) {
        $this->rpcExceptions[] = $rpcException;
    }

    public function loadFromRpcFormat (RpcFormat $rpcFormat) {
        $this->resSerialize = $rpcFormat->format;
    }

    public function loadFromRpcResponseHeader (RpcResponseHeader $rpcResponseHeader) {
        $header = new ResHeader();
        $header->name = $rpcResponseHeader->name;
        $header->value = $rpcResponseHeader->value;
        $this->resHeaders[] = $header;
    }

    public function loadFromRpcAuthority (RpcAuthority $rpcAuthority) {
        $this->needAuthorities[] = $rpcAuthority->need;
    }

    public function loadFromRpcMessageHook (RpcMessageHook $rpcMessageHook) {
        // 兼容旧websocket申明
        if (strpos($rpcMessageHook->name, "ws.") === 0) {
            print "\n###!!!!! RpcMessageHook ws.* for websocket is DEPRECATED and will be removed in a future version. Use RpcWebsocket instead.\n";
            $this->wsNeedConnectMessage = true;
            $this->wsNeedDisconnectMessage = true;
            $this->wsRoom = substr($rpcMessageHook->name, 3);
        } else {
            $this->messageHookName = $rpcMessageHook->name;
            $this->messageHookDedicatedPoolSize = $rpcMessageHook->dedicatedPoolSize;
        }
    }

    public function loadFromRpcWebsocket (RpcWebsocket $annotation) {
        $this->wsNeedConnectMessage = $annotation->needConnectMessage;
        $this->wsNeedDisconnectMessage = $annotation->needDisconnectMessage;
        $this->wsRoom = $annotation->roomName;
    }

    public function loadFromRpcDisableTypeValidation (RpcDisableTypeValidation $annotation) {
        $this->checkType = $annotation->check_type;
        $this->checkValidation = $annotation->check_validation;
    }

    public function loadFromRpcCron (RpcCron $annotation, $classAliasName) {
        $annotation->every and $this->cronSpec = '@every '.$annotation->every;
        $annotation->entry and $this->cronSpec = '@'.$annotation->entry;
        $annotation->spec and $this->cronSpec = str_replace('\\', '/', $annotation->spec);

        $this->cronHookName = "{$classAliasName}::{$this->name}";
    }

    public function loadFromRpcParam (RpcParam $rpcParam) {
        if ($this->requestType == RpcDefinition::EmptyType) {
            $this->requestType = sprintf("%s@%sParams", $this->controllerName, Str::studly($this->name));
        }

        $struct = RpcDefinition::getInstance()->notDefinitionStruct($this->requestType);
        $property = new Property();

        $names = explode('.', $rpcParam->name);
        $prefixType = $this->requestType;
        $childProperty = null;
        foreach ($names as $index => $child) {
            if ($child === "*") {
                if (!Str::endsWith($childProperty->type, '[]')) {
                    $childProperty->type .= '[]';
                }
                continue;
            }
            if ($index == count($names) - 1) {
                $property->require = $rpcParam->require;
                $property->validation = $rpcParam->validation;
                if (is_subclass_of($rpcParam->type, RpcObject::class)) {
                    $_ = RpcDefinition::getInstance()->objectStruct($rpcParam->type);
                }

                $property->type = Language::$rpcSimpleTypeMap[$rpcParam->type] ?? $rpcParam->type;
                $property->nullable = $rpcParam->nullable;
                $property->name = $child;
                $property->setSerializedName($child);

                if ($rpcParam->default !== null) {
                    $value = $rpcParam->default;
                    Language::tryAutoFixType($value, $property->type);

                    $property->default = $value;
                    $property->defaultValueAvailable = true;
                }
                $property->comment = $rpcParam->comment;
                $property->example = $rpcParam->example;
                $property->fromSession = $rpcParam->fromSession;
                $property->fromHeader = $rpcParam->fromHeader;
                if ($property->fromHeader) {
                    $header = new ReqHeader();
                    $header->name = $property->fromHeader;
                    $this->reqHeaders[$header->name] = $header;
                }

                $struct->properties[] = $property;
            } else {
                $type = $prefixType.Str::studly($child);
                $childProperty = $this->findProperty($struct->properties, $child);
                if (!$childProperty) {
                    $childStruct = RpcDefinition::getInstance()->notDefinitionStruct($type);
                    $childStruct->originName = implode('.', array_slice($names, 0, $index + 1));
                    $childProperty = new Property();
                    $childProperty->name = $child;
                    $childProperty->setSerializedName($child);
                    $struct->properties[] = $childProperty;
                    $struct = $childStruct;
                } else {
                    if (Str::endsWith($type, '[]')) {
                        $struct = RpcDefinition::getInstance()->notDefinitionStruct(substr($type, 0, -2));
                    } else {
                        $struct = RpcDefinition::getInstance()->notDefinitionStruct($type);
                    }
                }

                $childProperty->type = $type;
                $prefixType = $type;
            }
        }
    }

    private function findProperty ($properties, $name): ?Property {
        /** @var Property[] $properties */
        foreach ($properties as $property) {
            if ($property->name == $name) {
                return $property;
            }
        }

        return null;
    }

    /**
     * @return array
     */
    public function getRequestDeclareNames (): array {
        if ($this->requestDeclareNames === null) {
            $this->requestDeclareNames = [];
            foreach ($this->getRequestDeclareMap() as $name => $property) {
                $this->resolvePropertyNames($this->requestDeclareNames, "", $property);
            }
        }

        return $this->requestDeclareNames;
    }

    private function resolvePropertyNames (&$names, $prefix, Property $property) {
        $name = $prefix ? $prefix.'.'.$property->name : $property->name;

        if (RpcDefinition::getInstance()->inStructCache($property->type)) {
            foreach (RpcDefinition::getInstance()->getStruct($property->type)->properties as $childProperty) {
                $this->resolvePropertyNames($names, $name, $childProperty);
            }
        } else {
            $names[] = $name;
        }
    }

    /**
     * @return ReflectionMethod
     */
    public function getRefInstance (): ?ReflectionMethod {
        return $this->refInstance;
    }

    public function loadFromAlias (Alias $annotation) {
        $this->alias = $annotation->name;
    }

    public function loadFromRpcSupportEnv (RpcSupportEnv $annotation) {
        if ($annotation->enable && $annotation->disable) {
            throw new RpcRuntimeException("please use one of enable and disable");
        }

        if ($annotation->enable) {
            $this->enableEnv = $annotation->enable;
        }

        if ($annotation->disable) {
            $this->enableEnv = array_diff(RpcEnvironmentEnum::All, $annotation->disable);
        }
    }

    public function loadFromRpcDeprecated (RpcDeprecated $aDeprecated) {
        $deprecated = new Deprecated();
        $deprecated->reason = $aDeprecated->reason;
        $deprecated->since = $aDeprecated->since;
        $deprecated->replacement = $aDeprecated->replacement;
        $this->deprecated = $deprecated;
    }

    /**
     * handler doc, deprecated, return type
     *
     * @param string $docComment
     */
    public function loadDocument (string $docComment) {
        $this->description = "";

        foreach (explode("\n", $docComment) as $line) {
            if (strpos($line, '@') === false && strpos($line, '/') === false) {
                $_doc = trim(str_replace('*', '', $line));
                if (!$_doc) continue;

                $this->description .= $_doc."\n";
                continue;
            }


            if (strpos($line, '@since') !== false) {
                $this->since = trim(substr($line, strpos($line, '@since') + 6));
            }
        }

        $this->description = rtrim($this->description, "\n");
    }

    public function toArray (): array {
        return to_array(get_object_public_vars($this));
    }

    public function resolveReturnType () {
        if ($this->refInstance->hasReturnType()) {
            $useClasses = ClassRef::parseUseClasses($this->refInstance->getFileName());
            $doc = $this->refInstance->getDocComment();

            switch ($this->refInstance->getReturnType()->getName()) {
                case 'array':
                    [$this->responseType, $childType] = ArrayTypeParser::parseArrayReturn($useClasses, $doc, $this->refInstance->getNamespaceName());
                    $this->resolveChildType($childType);
                    break;
                case Map::class:
                    [$this->responseType, $childType] = MapTypeParser::parseMapReturn($useClasses, $doc, $this->refInstance->getNamespaceName());
                    $this->resolveChildType($childType);
                    break;
                default:
                    if (isset(Language::$simpleReturnTypeMap[$this->refInstance->getReturnType()->getName()])) {
                        $this->responseType = Language::$simpleReturnTypeMap[$this->refInstance->getReturnType()->getName()];
                    } else {
                        if (is_subclass_of($this->refInstance->getReturnType()->getName(), RpcObject::class)) {
                            RpcDefinition::getInstance()->objectStruct($this->refInstance->getReturnType()->getName());
                            $this->responseType = $this->refInstance->getReturnType()->getName();
                        } else {
                            $this->responseType = "mixed";
                        }
                    }
            }
        }
    }

    private function resolveChildType (?string $childType) {
        if (!$childType || in_array($childType, Language::$simpleType)) {
            return;
        }
        if (is_subclass_of($childType, RpcObject::class)) {
            if (!RpcDefinition::getInstance()->inStructCache($childType)) {
                $_ = RpcDefinition::getInstance()->objectStruct($childType);
            }
        } else {
            throw new RpcRuntimeException("property object type is not subclass of RpcObject : {$childType}");
        }
    }
}