<?php

namespace Mainto\RpcServer\Tracing\Extend\GuzzleHttp;

use Illuminate\Support\Str;
use Mainto\RpcServer\RpcServer\RpcContext;
use Mainto\RpcServer\RpcServer\RpcInvoke;
use Mainto\RpcServer\RpcUtil\JsonHelper;
use Mainto\RpcServer\Tracing\Context;
use Mainto\RpcServer\Tracing\Interfaces\AttributeKey;
use Mainto\RpcServer\Tracing\Interfaces\SpanKind;
use Mainto\RpcServer\Tracing\Span;
use Psr\Http\Message\RequestInterface;
use Psr\Http\Message\ResponseInterface;

/**
 * 拦截器对象
 */
class Middleware {
    /**
     * 返回请求链路拦截
     * @return callable
     */
    public static function traceMiddleware (): callable {
        return static function (callable $handler): callable {
            $ctx = Context::getSpanContext();
            return static function (RequestInterface $request, array $options) use ($handler, $ctx) {
                $spanName = sprintf(
                    '%s %s%s',
                    $request->getMethod(),
                    $request->getUri()->getHost(),
                    $request->getUri()->getPath(),
                );
                $isUtf8 = self::isUtf8Charset($request);
                $span = Span::start($spanName, $ctx, SpanKind::KIND_CLIENT);
                $span->setAttribute(AttributeKey::HTTPHeadersKey, JsonHelper::encodeWithUnescapedAndError($request->getHeaders()));
                if ($uri = $request->getUri()) {
                    $span->setAttribute(AttributeKey::HTTPUrlKey, $uri);
                    $span->setAttribute(AttributeKey::HTTPTargetKey, $uri->getHost());
                }
                if ($content = $request->getBody()->getContents()) {
                    $span->setAttribute(AttributeKey::HTTPBodyKey, $isUtf8 ? $content : base64_encode($content));
                }
                // 增加链路header，通过http传递链路信息
                $request = $request->withHeader(Context::TRACE_PARENT_HEADER, Context::getHeaderValue());
                // 增加泳道header，通过http传递灰度泳道标识
                $request = $request->withHeader("X-Mainto-Swimlane", RpcInvoke::getCurrentContext()->getSwimlane());
                // 后续执行
                return $handler($request, $options)->then(
                    static function (ResponseInterface $response) use ($request, $span, $isUtf8) {
                        $span->setAttribute(AttributeKey::HTTPStatusCodeKey, $response->getStatusCode());
                        $span->setAttribute(
                            'response.headers',
                            JsonHelper::encodeWithUnescapedAndError($response->getHeaders())
                        );
                        if ($content = $response->getBody()->getContents()) {
                            $span->setAttribute('response.body', $isUtf8 ? $content : base64_encode($content));
                        }
                        $response->getBody()->rewind();
                        if ($response->getStatusCode() >= 400) {
                            $span->statusError('request error!');
                        }
                        $span->end();
                        return $response;
                    }
                );
            };
        };
    }

    /**
     * @param RequestInterface $request
     * @return bool
     */
    private static function isUtf8Charset (RequestInterface $request): bool {
        $contentType = $request->getHeaderLine("Content-Type");
        if (empty($contentType)) {
            return true;
        }
        $contentType = strtolower($contentType);
        if (Str::contains($contentType, 'charset=utf-8')
            || !Str::contains($contentType, 'charset=')
        ) {
            return true;
        }
        return false;
    }
}