<?php


namespace Mainto\RpcServer\RpcUtil;


use Exception;
use Illuminate\Support\Facades\Log;
use Mainto\RpcServer\Base\RpcWsMessage;
use Mainto\RpcServer\Exceptions\RpcRuntimeException;
use Mainto\RpcServer\Protocol\Common\Body;
use Mainto\RpcServer\Protocol\Common\Types\RequestHeaderType;
use Mainto\RpcServer\Protocol\Common\Types\ResponseHeaderType;
use Mainto\RpcServer\Protocol\Request\Extend\RequestExtendHeader;
use Mainto\RpcServer\Protocol\Request\Request;
use Mainto\RpcServer\Protocol\Response\Extend\ResponseExtendError;
use Mainto\RpcServer\RpcClient\RpcClient;
use Mainto\RpcServer\RpcClient\RpcClientFactory;
use Mainto\RpcServer\RpcClient\RpcClientWeakRef;
use Mainto\RpcServer\RpcServer\RpcInvoke;
use Mainto\RpcServer\Util\Net\Exceptions\ServerNotFoundException;
use Ramsey\Uuid\Uuid;
use Ramsey\Uuid\UuidInterface;
use RuntimeException;
use Throwable;

class RpcMessage {
    private const WebsocketQueue = "micro-ws-send-queue";
    private const MessageQueue = "micro-msg-main-queue";

    public const PERSISTENT = 'persistent';
    public const TRANSIENT = 'transient';

    private static ?RpcMessage $instance = null;

    private string $serviceName = "";

    /**
     * @var RpcClientWeakRef
     */
    private RpcClientWeakRef $messageClientRef;

    private function __construct () {
    }

    public static function getInstance (): RpcMessage {
        if (!self::$instance) {
            $instance = new RpcMessage();
            $instance->messageClientRef = new RpcClientWeakRef([RpcClientFactory::MessageClient]);
            $instance->serviceName = config('rpc-server.service_name');

            self::$instance = $instance;
        }

        return self::$instance;
    }

    /**
     * @return RpcClient
     */
    protected function getClient (): RpcClient {
        return $this->messageClientRef->getClient();
    }

    public function clearClient () {
        $this->messageClientRef->clearClient();
    }

    private function mustGetUuid4 (): UuidInterface {
        try {
            return Uuid::uuid4();
        } catch (Exception $e) {
            throw new RuntimeException($e->getCode(), $e->getMessage());
        }
    }

    /**
     * 发送基础消息
     * @param \Mainto\RpcServer\Base\RpcMessage $message
     * @param string $mode 持久化模式 [默认持久化]
     */
    public function sendMessage (\Mainto\RpcServer\Base\RpcMessage $message, string $mode = self::PERSISTENT) {
        $request = new Request();
        $ext = new RequestExtendHeader();
        $ext->addHeader("content_type", "text/json");
        $ext->addHeader("key", self::MessageQueue);
        $ext->addHeader("app_id", $this->serviceName);
        $ext->addHeader("type", sprintf($message::messageName(), ...$message->formatValue()));
        $ext->addHeader("delivery_mode", $mode);
        $ext->addHeader("mq_headers", json_encode([
            "trace-id" => RpcInvoke::getCurrentTraceId(),
            "swimlane" => RpcInvoke::getCurrentSwimlane(),
        ]));

        $request->setTypeWithExtend(RequestHeaderType::MessageNormalType, $ext);
        $request->setBody(Body::newJsonBody(json_encode($message)));

        $this->doRequest($request);
    }

    /**
     * 发送消息到 WebSocket
     * @param string $roomName $message
     * @param string $clientId 接受客户端ID
     * @param RpcWsMessage $message
     * @param string $mode     持久化模式 [默认非持久化]
     */
    public function sendWsMessage (string $roomName, string $clientId, RpcWsMessage $message, string $mode = self::PERSISTENT) {
        $request = new Request();
        $ext = new RequestExtendHeader();
        $ext->addHeader("content_type", "text/json");
        $ext->addHeader("app_id", $this->serviceName);
        $ext->addHeader("delivery_mode", $mode);
        $ext->addHeader("key", self::WebsocketQueue);
        $ext->addHeader("mq_headers", json_encode([
            "room-name" => $roomName,
            "user-id"   => $clientId,
            "trace-id"  => RpcInvoke::getCurrentTraceId(),
            "swimlane"  => RpcInvoke::getCurrentSwimlane(),
        ]));

        $request->setTypeWithExtend(RequestHeaderType::WebsocketNormalType, $ext);
        $request->setBody(Body::newJsonBody(json_encode($message)));

        $this->doRequest($request);
    }

    /**
     * 发送延迟消息
     * @param int $delayTime 延迟时间 [单位: ms]
     * @param \Mainto\RpcServer\Base\RpcMessage $message
     * @param string $mode   持久化模式 [默认持久化]
     */
    public function sendDelayMessage (int $delayTime, \Mainto\RpcServer\Base\RpcMessage $message, string $mode = self::PERSISTENT) {
        $request = new Request();
        $ext = new RequestExtendHeader();
        $ext->addHeader("content_type", "text/json");
        $ext->addHeader("key", self::MessageQueue);
        $ext->addHeader("app_id", $this->serviceName);
        $ext->addHeader("type", $message->messageName());
        $ext->addHeader("delivery_mode", $mode);
        $ext->addHeader("mq_headers", json_encode([
            "delay-time" => $delayTime,
            "trace-id"   => RpcInvoke::getCurrentTraceId(),
            "swimlane"   => RpcInvoke::getCurrentSwimlane(),
        ]));

        $request->setTypeWithExtend(RequestHeaderType::MessageDelayType, $ext);
        $request->setBody(Body::newJsonBody(json_encode($message)));

        $this->doRequest($request);
    }

    /**
     * @param Request $request
     */
    private function doRequest (Request $request): void {
        try {
            $response = $this->getClient()->Do($request);
            switch ($response->getType()) {
                case ResponseHeaderType::ReturnOKType:
                    break;
                case ResponseHeaderType::ReturnSystemErrType:
                case ResponseHeaderType::ReturnErrType:
                    $extend = $response->getExt();
                    if ($extend instanceof ResponseExtendError) {
                        throw new RpcRuntimeException($extend->getErrMsg(), $extend->getErrorCode());
                    }

                    throw new RuntimeException("unknown response ext");
                default:
                    throw new RuntimeException("unknown response");
            }
        } catch (Throwable $exception) {
            if ($exception instanceof RpcRuntimeException) {
                throw $exception;
            }
            $this->clearClient();
            Log::error($exception);

            if ($exception instanceof ServerNotFoundException) {
                throw new RuntimeException("can not find message queue", 0, $exception);
            }
            throw new RuntimeException("can not connect to message queue", 0, $exception);
        }
    }

    public static function getMessageQueue (): string {
        return self::MessageQueue;
    }

    public static function getWebsocketQueue (): string {
        return self::WebsocketQueue;
    }
}