<?php


namespace Mainto\RpcServer\RpcServer\Middleware\Kernel;


use Closure;
use Exception;
use Illuminate\Contracts\Debug\ExceptionHandler;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use Mainto\RpcServer\Exceptions\BaseNotReportServiceException;
use Mainto\RpcServer\Exceptions\BaseServiceException;
use Mainto\RpcServer\Protocol\Common\Body;
use Mainto\RpcServer\Protocol\Response\Extend\ResponseExtendError;
use Mainto\RpcServer\Protocol\Response\Response;
use Mainto\RpcServer\RpcServer\Middleware\KernelMiddleware;
use Mainto\RpcServer\RpcServer\RpcResponseHelper;
use Mainto\RpcServer\RpcServer\RpcStreamContext;
use Mainto\RpcServer\RpcUtil\Tool\RpcLog;
use Psr\SimpleCache\InvalidArgumentException;
use Throwable;

class ProtectedKernelMiddleware implements KernelMiddleware {

    /**
     * @param RpcStreamContext $context
     * @param Closure $next
     * @return Response
     */
    public function handle (RpcStreamContext $context, Closure $next): Response {
        try {
            return $next($context);
        } catch (Throwable $exception) {
            /** @var ResponseExtendError $errExt */
            $cacheErrResponse = RpcResponseHelper::$cacheResponse->resetBodyAndExt()->useReturnErrType();
            $cacheExtendError = RpcResponseHelper::$cacheExtendError;
            $cacheErrResponse->setExtend($cacheExtendError);
            $cacheExtendError->setTraceId($context->request->getTraceId());
            $cacheExtendError->setErrMsg($exception->getMessage());
            $cacheExtendError->switchNormalError();

            if ($exception instanceof ValidationException) {
                $cacheExtendError->setErrorCode(422);
                $cacheExtendError->setErrMsg(json_encode($exception->errors()));
            } elseif ($exception instanceof InvalidArgumentException) {
                $cacheExtendError->setErrorCode(500);
            } else {
                if ($exception instanceof BaseServiceException) {
                    if ($exception->hasBody()) {
                        $encode = body_encode($exception->getBody());
                        if (strlen($encode) < 1024 * 1024) {
                            $cacheErrResponse->setBody(Body::newJsonBody($encode));
                        }
                    }
                    $cacheExtendError->switchServiceError();
                }
                if (!($exception instanceof BaseNotReportServiceException)) {
                    try {
                        RpcLog::getInstance()->logRequestThrowable($context->request, $exception, "{$context->request->getCallClassName()}::{$context->request->getCallMethodName()}");
                    } catch (Throwable $e) {
                        Log::error($e, ["streamId" => $context->streamId]);
                        // nop
                    }
                } else {
                    $cacheExtendError->switchServiceNotReportError();
                }

                $cacheExtendError->setErrorCode(intval($exception->getCode()) ?: 500);
            }

            if ($exception instanceof Exception) {
                app(ExceptionHandler::class)->report($exception);
            } else {
                Log::error($exception, ["streamId" => $context->streamId]);
            }

            return $cacheErrResponse;
        }
    }
}
