<?php

namespace Mainto\RpcServer\Tracing\Extend\GuzzleHttp;

use Mainto\RpcServer\RpcUtil\JsonHelper;
use Mainto\RpcServer\Tracing\Context;
use Mainto\RpcServer\Tracing\Interfaces\AttributeKey;
use Mainto\RpcServer\Tracing\Interfaces\SpanKind;
use Mainto\RpcServer\Tracing\Span;
use Psr\Http\Message\RequestInterface;
use Psr\Http\Message\ResponseInterface;

/**
 * 拦截器对象
 */
class Middleware {
    /**
     * 返回请求链路拦截
     * @return callable
     */
    public static function traceMiddleware (): callable {
        return static function (callable $handler): callable {
            $ctx = Context::getSpanContext();
            return static function (RequestInterface $request, array $options) use ($handler, $ctx) {
                $spanName = sprintf(
                    '%s %s%s',
                    $request->getMethod(),
                    $request->getUri()->getHost(),
                    $request->getUri()->getPath(),
                );
                $bodyEncode = self::checkEncodeWithCharset($request);
                $span = Span::start($spanName, $ctx, SpanKind::KIND_CLIENT);
                $span->setAttribute(AttributeKey::HTTPHeadersKey, JsonHelper::encodeWithUnescapedAndError($request->getHeaders()));
                if ($uri = $request->getUri()) {
                    $span->setAttribute(AttributeKey::HTTPUrlKey, $uri);
                    $span->setAttribute(AttributeKey::HTTPTargetKey, $request->getUri()->getHost());
                }
                if ($content = $request->getBody()->getContents()) {
                    $span->setAttribute(AttributeKey::HTTPBodyKey, $bodyEncode ? base64_encode($content) : $content);
                }
                // 增加链路header，通过http传递链路信息
                $request = $request->withHeader(Context::TRACE_PARENT_HEADER, Context::getHeaderValue());
                // 后续执行
                return $handler($request, $options)->then(
                    static function (ResponseInterface $response) use ($request, $span, $bodyEncode) {
                        $span->setAttribute(AttributeKey::HTTPStatusCodeKey, $response->getStatusCode());
                        $span->setAttribute(
                            'response.headers',
                            JsonHelper::encodeWithUnescapedAndError($response->getHeaders())
                        );
                        if ($content = $response->getBody()->getContents()) {
                            $span->setAttribute('response.body', $bodyEncode ? base64_encode($content) : $content);
                        }
                        $response->getBody()->rewind();
                        if ($response->getStatusCode() >= 400) {
                            $span->statusError('request error!');
                        }
                        $span->end();
                        return $response;
                    }
                );
            };
        };
    }

    /**
     * @param RequestInterface $request
     * @return bool
     */
    private static function checkEncodeWithCharset (RequestInterface $request): bool {
        $contentType = $request->getHeaderLine("Content-Type");
        if (empty($contentType)) {
            return false;
        }
        $types = explode('; ', $contentType);
        foreach ($types as $type) {
            if (strpos($type, '=') > 0) {
                [$key, $value] = explode('=', strtolower($type));
                if ($key === 'charset' && $value === 'gbk') {
                    return true;
                }
            }
        }
        return false;
    }
}