<?php

namespace Mainto\RpcServer\Command;

use Doctrine\Common\Annotations\AnnotationReader;
use Illuminate\Console\Command;
use Illuminate\Support\Str;
use Mainto\RpcServer\Base\RpcEnum;
use Mainto\RpcServer\Exceptions\BaseServiceException;
use Mainto\RpcServer\Exceptions\RpcRuntimeException;
use Mainto\RpcServer\RpcAnnotations\Alias;
use Mainto\RpcServer\RpcServer\Definition\Controller;
use Mainto\RpcServer\RpcServer\Definition\Deprecated;
use Mainto\RpcServer\RpcServer\Definition\Method;
use Mainto\RpcServer\RpcServer\RpcDefinition;
use Mainto\RpcServer\RpcServer\RpcRouter;
use Mainto\RpcServer\RpcServer\Definition\Enum;
use Mainto\RpcServer\RpcServer\Definition\EnumUnit;
use Mainto\RpcServer\RpcServer\Definition\Exception;
use Mainto\RpcServer\RpcServer\Definition\ExceptionCode;
use ReflectionClass;
use RuntimeException;

/**
 * Class RpcRegisterCommand
 * @package Mainto\RpcServer\Command
 */
class RpcServiceInfoCommand extends Command {
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'rpc:service-info {option}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'RPC Register Info Command';
    private array $_exceptionCodes;

    public function handle () {
        switch ($this->argument('option')) {
            case 'controller':
                dump(RpcDefinition::getInstance()->controllers->mapWithKey(function (Controller $controller, $registerName) {
                    return [$registerName => (new ReflectionClass($controller->getInstance()))->getNamespaceName().'\\'.$controller->name];
                }));
                break;
            case 'url':
                $map = [];
                RpcDefinition::getInstance()->controllers->each(function (Controller $controller) use (&$map) {
                    $controller->methods->each(function (Method $method) use (&$map) {
                        if ($method->httpApi) {
                            $map[$method->httpApi->toString()] = $method->path();
                        }
                    });
                });
                dump($map);
                break;
            case 'cron':
                $map = [];
                RpcDefinition::getInstance()->controllers->each(function (Controller $controller) use (&$map) {
                    $controller->methods->each(function (Method $method) use (&$map) {
                        if ($method->cronHookName) {
                            $map[$method->cronSpec] = $method->path();
                        }
                    });
                });
                dump($map);
                break;
            case 'version':
                dump(RpcDefinition::getInstance()->allVersion());
                break;
            case 'router':
                $map = [];
                RpcDefinition::getInstance()->controllers->each(function (Controller $controller) use (&$map) {
                    $controller->methods->each(function (Method $method) use ($controller, &$map) {
                        $map[$controller->registerName][] = $method->path();
                    });
                });
                dump($map);
                break;
            case 'dump':
                $serviceInfo = base_path("service_info");
                file_exists($serviceInfo) && remove_all($serviceInfo);
                @mkdir($serviceInfo);
                if (file_exists(base_path("doc"))) {
                    copy_dir(base_path("doc"), path_join($serviceInfo, "doc"));
                }

                [$enums, $exceptions] = $this->resolveEnumAndException();

                $definition = RpcDefinition::getInstance();

                if (file_exists(base_path('request_examples.json'))) {
                    $requestExamples = json_decode(file_get_contents(base_path('request_examples.json')), true);
                    foreach ($requestExamples as $registerName => $methods) {
                        foreach ($methods as $methodName => $examples) {
                            foreach ($examples as $title => $example) {
                                $definition->controllers[$registerName]->methods[$methodName]->requestExamples[$title] = $example;
                            }
                        }
                    }
                }

                file_put_contents(path_join($serviceInfo, "definition.json"), json_encode(array_merge(
                    $definition->toArray(),
                    [
                        'dependencies' => $this->getDependencies(),
                        'enums'        => $enums,
                        'exceptions'   => $exceptions,
                    ]
                ), JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));

        }
    }

    public function resolveEnumAndException (): array {
        $appDir = app_path();
        if (RpcRouter::getAppDir()) {
            $appDir = RpcRouter::getAppDir();
        }

        $classes = get_classes($appDir);
        $refEnums = $refExceptions = [];
        foreach ($classes as $class) {
            $ref = new ReflectionClass($class);
            if ($ref->isSubclassOf(RpcEnum::class)) {
                $refEnums[] = $ref;
            }

            if ($ref->isSubclassOf(BaseServiceException::class)) {
                $refExceptions[] = $ref;
            }
        }

        $enums = [];
        foreach ($refEnums as $refEnum) {
            $enums[] = $this->resolveEnum($refEnum);
        }

        $exceptions = [];
        foreach ($refExceptions as $refException) {
            $exception = $this->resolveException($refException);
            if ($exception) {
                $exceptions[] = $exception;
            }
        }

        return [$enums, $exceptions];
    }

    /**
     * @param ReflectionClass $reflection
     * @return Enum
     */
    private function resolveEnum (ReflectionClass $reflection): Enum {
        $enum = new Enum();

        $currentName = preg_replace('/.*[aA]pp\\\\[\w]+\\\\?/', '', $reflection->getNamespaceName());
        $currentName = preg_replace('/\\\\?(Enums|Enum)\\\\?/', '\\', $currentName);
        $simpleNamespace = trim($currentName, '\\');

        $enum->namespace = $simpleNamespace;
        $enum->name = class_basename($reflection->getName());
        $reflectionClassConstants = $reflection->getReflectionConstants();
        foreach ($reflectionClassConstants as $classConstant) {
            $unit = new EnumUnit();
            $unit->name = $classConstant->name;
            $unit->value = $classConstant->getValue();
            $unit->comment = get_doc($classConstant->getDocComment());
            $enum->enums[] = $unit;
        }
        $annotations = app(AnnotationReader::class)->getClassAnnotations($reflection);
        foreach ($annotations as $annotation) {
            if ($annotation instanceof Alias) {
                $enum->alias = $annotation->name;
            }
        }

        $description = "";
        foreach (explode("\n", $reflection->getDocComment()) as $line) {
            if (strpos($line, '@') === false && strpos($line, '/') === false) {
                $_doc = trim(str_replace('*', '', $line));
                if (!$_doc) continue;

                $description .= $_doc."\n";
                continue;
            }
            if (strpos($line, '@deprecated') !== false) {
                $deprecated = new Deprecated();
                $deprecated->reason = trim(substr($line, strpos($line, '@deprecated') + 11));
                $enum->deprecated = $deprecated;
            }

            if (strpos($line, '@since') !== false) {
                $enum->since = trim(substr($line, strpos($line, '@since') + 6));
            }
        }

        $enum->description = rtrim($description, "\n");

        return $enum;
    }

    /**
     * @param ReflectionClass $reflection
     * @return Exception|void
     */
    private function resolveException (ReflectionClass $reflection) {
        $regex = '/@method[ ]*static[ ]*([A-Z0-9a-z_]?[A-Za-z0-9_]*)[ ]*([A-Z0-9_]*)[ ]*\([ ]*\$(code|codeOrText)[ ]*=[ ]*(0x[0-9A-F]{9})[ ]*,[ ]*\$(text)[ ]*=[ ]*[\'"](.*)[\'"][ ]*\)/m';
        if ($reflection->isSubclassOf(BaseServiceException::class)) {
            $comment = '';

            $currentName = preg_replace('/.*[aA]pp\\\\[\w]+\\\\?/', '', $reflection->getName());
            $currentName = preg_replace('/\\\\?(Exceptions|Exception)\\\\/', '\\', $currentName);
            $simpleNamespace = trim($currentName, '\\');
            $exceptionClassName = $alias = $simpleNamespace;

            $annotations = app(AnnotationReader::class)->getClassAnnotations($reflection);
            foreach ($annotations as $annotation) {
                if ($annotation instanceof Alias) {
                    $alias = $annotation->name;
                }
            }

            $exception = new Exception();
            $exception->alias = $alias;
            $exception->name = $exceptionClassName;
            $exception->codes = [];

            foreach (explode("\n", $reflection->getDocComment()) as $line) {
                $line = trim($line, " \t\n\r\0\x0B*");

                if (strpos($line, '@method') !== false) {
                    if (preg_match_all($regex, $line, $matches, PREG_SET_ORDER, 0)) {
                        [$_, $exceptionName, $exceptionMethodName, $firstParamName, $exceptionCode, $secondParamName, $message] = $matches[0];
                        $code = intval(substr($exceptionCode, 2), 16);
                        if (isset($this->_exceptionCodes[$code])) {
                            throw new RpcRuntimeException("duplicate exception code on {$reflection->getName()}, code : $exceptionCode");
                        }

                        $this->_exceptionCodes[$code] = true;

                        if ($exceptionName != $reflection->getShortName()) {
                            throw new RuntimeException("doc: {$line} return name is not equals class {$reflection->getName()}", 500);
                        }

                        $exceptionCode = new ExceptionCode();
                        $exceptionCode->code = $code;
                        $exceptionCode->comment = $comment ?: $message;
                        $exceptionCode->message = $message;
                        $exceptionCode->exceptionName = $exceptionMethodName;

                        $exception->codes[] = $exceptionCode;
                        $comment = '';
                    } else {
                        throw new RpcRuntimeException("match failed on on {$reflection->getName()}, line content : $line");
                    }
                }
                if (strpos($line, '@exception-text') !== false) {
                    $comment = trim(str_replace(['@exception-text', '*'], '', $line));
                }
            }
            if (!$exception->codes) {
                return null;
            }

            return $exception;
        }
    }

    private function getDependencies (): array {
        $composeFile = base_path('composer.lock');

        if (file_exists($composeFile)) {
            $composeJson = json_decode(file_get_contents($composeFile), true);
            // substr mainto/micro-bridge-
            return array_values(array_map(function ($item) {
                return [
                    'name'      => Str::studly(substr($item['name'], 20)),
                    'version'   => $item['version'],
                    'reference' => $item['dist']['reference'],
                ];
            },
                array_filter($composeJson['packages'],
                    fn($item) => Str::startsWith($item['name'], "mainto/micro-bridge-")
                )
            ));
        } else {
            return [];
        }
    }
}