<?php

namespace Mainto\RpcTool\TestHelper;

use Doctrine\Common\Annotations\AnnotationReader;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;
use Mainto\RpcServer\Protocol\Common\Body;
use Mainto\RpcServer\Protocol\Common\Types\RequestHeaderType;
use Mainto\RpcServer\Protocol\Request\Extend\RequestExtendUrlInvoke;
use Mainto\RpcServer\Protocol\Request\Request;
use Mainto\RpcServer\RpcAnnotations\Alias;
use Mainto\RpcServer\RpcServer\RpcInvoke;
use Mainto\RpcServer\RpcServer\RpcObject;
use Mainto\RpcServer\RpcServer\RpcRouter;
use ReflectionException;
use ReflectionMethod;
use RuntimeException;
use Throwable;

trait ControllerDoc {
    /**
     * @param $class
     * @param $method
     * @param $params
     * @return mixed
     */
    public function doInvoke ($class, $method, $params) {
        if ($params instanceof RpcObject) {
            $params = $params->jsonSerialize();
        }

        $request = new Request();
        $request->setType(RequestHeaderType::InvokeNormalType);
        $request->setCallClassAndMethod($class, $method);
        $request->setBody(Body::newJsonBody(json_encode($params)));

        $requestExtendUrlInvoke = new RequestExtendUrlInvoke();
        $requestExtendUrlInvoke->addHeader("X-Stream-Id", "unit_session_id");

        $request->setExtend($requestExtendUrlInvoke);

        return RpcInvoke::invokeThroughMiddleware(RpcInvoke::newRpcContext($request));
    }

    /**
     * the controller class eg : ExampleController::class
     *
     * @return string
     */
    protected function getClass () { }

    /**
     * @return string
     */
    protected function getClassCallName (): string {
        $alias = RpcRouter::getInstance()->getControllerRegisterName($this->getClass());
        if ($alias) {
            return $alias;
        } else {
            throw new RuntimeException('can not auto find class alias name, please implement me!');
        }
    }

    public static function enableDump () {
        return config('rpc-server.test.dump_request_example');
    }

    /**
     * @param $name
     * @param $arguments
     * @return mixed
     * @throws Throwable
     * @throws ValidationException
     */
    public function __call ($name, $arguments) {
        if (strpos($name, 'call') === 0) {
            $method = Str::camel(substr($name, 4));
            if (method_exists($this->getClass(), $method)) {
                if (self::enableDump()) {
                    return $this->wrapInvoke($this->getClassCallName(), $method, ...$arguments);
                } else {
                    return $this->doInvoke($this->getClassCallName(), $method, ...$arguments);
                }
            }
        }
        throw new \BadMethodCallException($name);
    }

    public function wrapInvoke ($class, $method, $params) {
        $refTestMethod = $this->getCalledTestMethod(get_called_class(), debug_backtrace(false, 10));

        [$title, $description] = $this->getTestMethodTitleAndDescription($refTestMethod);

        $example = [
            "comment"  => $description,
            "request"  => $params,
            "response" => "",
            "name"     => $title,
        ];

        try {
            $response = $this->doInvoke($class, $method, $params);

            if (RpcInvoke::getCurrentContext() != null
                && RpcInvoke::getCurrentContext()->getMethod()->resSerialize == 'Raw'
                && preg_match('/[^\x20-\x7E\t\r\n\x{4e00}-\x{9fa5}]/mu', $response) !== 0) {
                $response = 'Raw';
            }

            $example['response'] = [
                "success" => true,
                "msg"     => $response,
            ];

            return $response;
        } catch (Throwable $e) {
            $example['response'] = [
                "success"    => false,
                "error_code" => $e->getCode(),
                "error_msg"  => $e instanceof ValidationException ? $e->errors() : $e->getMessage(),
            ];
            throw $e;
        } finally {
            $registerName = $class;
            $examplesPath = base_path("request_examples.json");
            if (file_exists($examplesPath)) {
                $content = json_decode(file_get_contents($examplesPath), true);
                $content[$registerName][$method][$title] = $example;
            } else {
                $content = [
                    $registerName => [
                        $method => [
                            $title => $example
                        ]
                    ]
                ];
            }
            file_put_contents($examplesPath, json_encode($content, JSON_UNESCAPED_UNICODE));
        }
    }

    /**
     * 获取调用 getResponse 的对应测试function 的ReflectionMethod实例
     *
     * @param $called_class
     * @param $debug_backtrace
     * @return ReflectionMethod
     * @throws ReflectionException
     */
    private function getCalledTestMethod ($called_class, $debug_backtrace): ReflectionMethod {
        $begin = false;
        $called_method_name = '';
        foreach ($debug_backtrace as $debug) {
            if (!$begin and $debug['class'] === $called_class
                && ($debug['function'] == "__call" || $debug['function'] == "callAndDoc")) {
                $begin = true;
                continue;
            }
            if ($debug['class'] === 'PHPUnit\Framework\TestCase') break;
            if ($begin) {
                $called_method_name = $debug['function'];
            }
        }
        return new ReflectionMethod($called_class, $called_method_name);
    }

    /**
     * 获取对应测试的方法注释  必须包含 Alias 用于请求示例的标题
     *
     * @param ReflectionMethod $refTestMethod
     * @return array
     */
    private function getTestMethodTitleAndDescription (ReflectionMethod $refTestMethod): array {
        $reader = new AnnotationReader();
        $annotations = $reader->getMethodAnnotations($refTestMethod);
        $test_title = '';
        foreach ($annotations as $annotation) {
            if ($annotation instanceof Alias) {
                $test_title = $annotation->name;
                break;
            }
        }
        if (!$test_title) {
            $test_title = $refTestMethod->getDeclaringClass()->getShortName()."::".$refTestMethod->name;
        }
        $test_description = '';
        $docComment = $refTestMethod->getDocComment();
        foreach (explode("\n", $docComment) as $line) {
            if (strpos($line, '@') === false && strpos($line, '/') === false) {
                $_doc = trim(str_replace('*', '', $line));
                if (!$_doc) continue;
                $test_description .= $_doc."\n";
            }
        }

        return [$test_title, trim($test_description)];
    }
}