<?php


namespace Mainto\RpcServer\RpcServer\Definition\RpcObject;


use Doctrine\Common\Annotations\AnnotationReader;
use Illuminate\Support\Str;
use Mainto\RpcServer\Exceptions\RpcRuntimeException;
use Mainto\RpcServer\RpcAnnotations\RpcComment;
use Mainto\RpcServer\RpcAnnotations\RpcExample;
use Mainto\RpcServer\RpcAnnotations\RpcProperty;
use Mainto\RpcServer\RpcAnnotations\RpcValidation;
use Mainto\RpcServer\RpcAnnotations\RpcValueFromHeader;
use Mainto\RpcServer\RpcAnnotations\RpcValueFromSession;
use Mainto\RpcServer\RpcAnnotations\Types\RpcArrayUint;
use Mainto\RpcServer\RpcAnnotations\Types\RpcUint;
use Mainto\RpcServer\RpcAnnotations\Types\RpcUnsignedDouble;
use Mainto\RpcServer\RpcServer\Definition\Property;
use Mainto\RpcServer\RpcServer\Definition\Traits\ValidateGetter;
use Mainto\RpcServer\RpcServer\RpcDefinition;
use Mainto\RpcServer\RpcServer\RpcObject;
use Mainto\RpcServer\Util\Language;
use Mainto\RpcServer\Util\ObjectMapper\ArrayTypeParser;
use Mainto\RpcServer\Util\ObjectMapper\ClassRef;
use Mainto\RpcServer\Util\ObjectMapper\MapTypeParser;
use Mainto\RpcServer\Util\Types\Map;
use ReflectionException;
use ReflectionMethod;
use ReflectionProperty;

class ObjectRef extends ClassRef {
    use ValidateGetter;

    /**
     * @var array
     */
    private static array $classRefMap = [];

    public static function clear () {
        self::$classRefMap = [];
    }

    /**
     * @param $object
     * @return ObjectRef
     * @throws ReflectionException
     */
    public static function getRef ($object): ObjectRef {
        if (is_object($object)) {
            return self::getRefByInstance($object);
        }
        $objectKey = $object;

        return static::$classRefMap[$objectKey] ?? (static::$classRefMap[$objectKey] = new static($object));
    }

    public static function getRefByInstance (object $instance): ObjectRef {
        $objectKey = get_class($instance);

        return static::$classRefMap[$objectKey] ?? (static::$classRefMap[$objectKey] = new static($instance));
    }

    /**
     * 分析类 public 属性
     */
    public function loadProperties () {
        $methods = array_flip(array_map(fn($method) => $method->name, $this->class->getMethods(ReflectionMethod::IS_PUBLIC)));

        // todo in php8 could use https://www.php.net/manual/en/reflectionproperty.getdefaultvalue.php
        $defaultProperties = $this->class->getDefaultProperties();

        $this->properties = array_map(function (ReflectionProperty $property) use ($methods, $defaultProperties) {
            $setMethod = 'set'.Str::studly($property->name);

            $definitionProperty = new Property();
            $definitionProperty->name = $property->name;
            [$definitionProperty->type, $childType] = $this->getType($property, $this);
            if ($childType && is_subclass_of($childType, RpcObject::class)) {
                if (!RpcDefinition::getInstance()->inStructCache($childType)) {
                    $_ = RpcDefinition::getInstance()->objectStruct($childType);
                }
            } else {
                if ($childType && !in_array($childType, Language::$simpleType)) {
                    throw new RpcRuntimeException("property object type is not subclass of RpcObject : {$childType}");
                }
            }

            if (optional($property->getType())->allowsNull()) {
                $definitionProperty->nullable = true;
            }

            foreach (app(AnnotationReader::class)->getPropertyAnnotations($property) as $annotation) {
                if ($annotation instanceof RpcComment) {
                    $definitionProperty->comment = $annotation->comment;
                } elseif ($annotation instanceof RpcValidation) {
                    $definitionProperty->validation = $annotation->validation;
                } elseif ($annotation instanceof RpcUint) {
                    $definitionProperty->type = 'uint';
                } elseif ($annotation instanceof RpcUnsignedDouble) {
                    $definitionProperty->type = 'unsignedDouble';
                } elseif ($annotation instanceof RpcArrayUint) {
                    $definitionProperty->type = 'uint[]';
                } elseif ($annotation instanceof RpcExample) {
                    $definitionProperty->example = $annotation->example;
                } elseif ($annotation instanceof RpcValueFromSession) {
                    $definitionProperty->fromSession = $annotation->key;
                } elseif ($annotation instanceof RpcValueFromHeader) {
                    $definitionProperty->fromHeader = $annotation->key;
                } elseif ($annotation instanceof RpcProperty) {
                    if ($annotation->name && $annotation->name != $definitionProperty->name) {
                        $definitionProperty->setSerializedName($property->name);
                        $definitionProperty->name = $annotation->name;
                    }
                    $annotation->type && $definitionProperty->type = $annotation->type;
                    $annotation->validation && $definitionProperty->validation = $annotation->validation;
                    $annotation->comment && $definitionProperty->comment = $annotation->comment;
                    $annotation->fromSession && $definitionProperty->fromSession = $annotation->fromSession;
                    $annotation->fromHeader && $definitionProperty->fromHeader = $annotation->fromHeader;
                    $annotation->example && $definitionProperty->example = $annotation->example;
                }
            }
            if ($definitionProperty->fromSession && !in_array($definitionProperty->type, Language::$rpcSimpleType)) {
                throw new RpcRuntimeException("from session property only support rpc simple type");
            }

            if (array_key_exists($property->name, $defaultProperties)) {
                $definitionProperty->default = $defaultProperties[$property->name];
                $definitionProperty->defaultValueAvailable = true;
            } else {
                $definitionProperty->require = true;
            }

            $definitionProperty->setterMethod = isset($methods[$setMethod]) ? $setMethod : '';

            return $definitionProperty;
        }, $this->class->getProperties(ReflectionProperty::IS_PUBLIC));
    }

    protected function getType (ReflectionProperty $property, ClassRef $classRef) {
        if ($property->getType()) {
            $type = $property->getType()->getName();
            switch ($type) {
                case 'array':
                    return ArrayTypeParser::parseArrayProperty($property->getDocComment(), $classRef->useClasses, $classRef->namespace);
                case Map::class:
                    return MapTypeParser::parseMapProperty($property->getDocComment(), $classRef->useClasses, $classRef->namespace);
                case 'float':
                    return ['double', null];
                default:
                    return [$type, $type];
            }
        }

        return ['mixed', null];
    }
}