<?php


namespace Mainto\RpcServer\Protocol\Common;


use Illuminate\Support\Facades\Log;
use Mainto\RpcServer\Exceptions\RpcRuntimeException;
use Mainto\RpcServer\Protocol\Common\Types\BodyType;
use Mainto\RpcServer\Util\Binary;
use Mainto\RpcServer\Util\Bytes\Buffer;
use Mainto\RpcServer\Util\Bytes\Bytes;
use Mainto\RpcServer\Util\IO\Closer;
use Mainto\RpcServer\Util\IO\IO;
use Mainto\RpcServer\Util\IO\IOUtil;
use Mainto\RpcServer\Util\IO\Reader;
use Mainto\RpcServer\Util\IO\ReaderFrom;
use Mainto\RpcServer\Util\IO\Writer;
use Mainto\RpcServer\Util\IO\WriterTo;
use Mainto\RpcServer\Util\Reader\LimitReader;
use RuntimeException;

class Body implements WriterTo, ReaderFrom {
    /**
     * uint8
     * @var int
     */
    private int $bodyType;

    /**
     * uint32
     * @var string
     */
    private string $bodySize;


    private ?Reader $bodyReader;

    /**
     * Body constructor.
     */
    private function __construct () {
    }

    public static function bodyFromReader (Reader $reader): Body {
        $object = new self();
        $object->readFrom($reader);

        return $object;
    }

    public function readFrom (Reader $reader): int {
        $read = $reader->read(5);
        $this->bodyType = Binary::strToUint8($read[0]);
        $this->bodySize = Binary::$littleEndian::strToUint32(substr($read, 1, 4));

        // todo 这里应该不需要全部读取出来
        $buffer = new Buffer();
        IO::copy($buffer, IO::limitReader($reader, $this->bodySize));

        $this->bodyReader = $buffer;

        return 5;
    }

    public static function newRawBody (string $body): Body {
        return self::newBody(BodyType::RAW, $body);
    }

    public static function newStreamBodyWithReader (int $bodySize, Reader $reader): Body {
        return self::newBodyWithReader(BodyType::RAW, $bodySize, $reader);
    }

    public static function newJsonBody (string $body): Body {
        return self::newBody(BodyType::JSON, $body);
    }

    public static function newBodyWithReader (int $bodyType, int $bodySize, Reader $reader): Body {
        $object = new self();
        $object->bodyType = $bodyType;
        $object->bodySize = $bodySize;
        $object->bodyReader = $reader;
        return $object;
    }

    public static function newBody (int $bodyType, string $body) {
        $object = new self();
        $object->bodyType = $bodyType;
        $object->bodySize = strlen($body);

        $buffer = new Buffer();
        $buffer->writeString($body);

        $object->bodyReader = $buffer;

        return $object;
    }

    public function setBodyBytes (string $body) {
        $this->bodySize = strlen($body);

        $this->bodyReader = new Buffer();
        $this->bodyReader->writeString($body);
    }

    public function reset () {
        $this->bodySize = 0;

        if ($this->bodyReader instanceof Closer) {
            $this->bodyReader->close();
        }

        $this->bodyReader = null;
    }

    public function cacheBody (): self {
        $buffer = Bytes::newBuffer();

        IO::copy($buffer, $this->getBodyReader());

        $this->bodyReader = $buffer;

        return $this;
    }

    public function getBodyReader (): Reader {
        return IO::limitReader($this->bodyReader, $this->bodySize);
    }

    public function getBodyBytes (bool $cache = false): string {
        $data = IOUtil::readAll($this->bodyReader);
        if ($cache) {
            $this->bodyReader = new Buffer();
            $this->bodyReader->write($data);
        }

        return $data;
    }

    public function encode () {
        return IO::IOWriterToBytes($this);
    }

    public function getContent () {
        switch ($this->bodyType) {
            case BodyType::JSON:
                $bodyBytes = $this->getBodyBytes(true);
                if ($bodyBytes === 'false') {
                    return false;
                }

                $decode = json_decode($bodyBytes, true);
                if ($decode === null && $bodyBytes !== "null") {
                    throw new RpcRuntimeException("body decode fail: invalid json");
                }

                return $decode;
            case BodyType::RAW:
                // 处理 body size
                if ($this->bodySize > 102400) {
                    return $this->bodyReader;
                }
                return $this->getBodyBytes(true);
            default:
                throw new RuntimeException("body type not support");
        }
    }

    public function contentLog() {
        switch ($this->bodyType) {
            case BodyType::JSON:
                $bodyBytes = $this->getBodyBytes(true);
                if ($bodyBytes === 'false') {
                    return false;
                }

                 $decode = json_decode($bodyBytes, true);
                if ($decode === null && $bodyBytes !== "null") {
                    throw new RpcRuntimeException("body decode fail: invalid json");
                }

                return $decode;
            case BodyType::RAW:
                // 处理 body size
                if ($this->bodySize > 102400) {
                    return "({$this->bodySize} byte binary)";
                }
                return $this->getBodyBytes(true);
            default:
                throw new RuntimeException("body type not support");
        }
    }

    public function toRaw (): string {
        switch ($this->bodyType) {
            case BodyType::RAW:
                return $this->getBodyBytes(true);
            case BodyType::JSON:
            default:
                throw new RuntimeException("body type not support");
        }
    }

    private function encodeSize (): string {
        $makeBytes = Binary::makeBytes(5);
        $makeBytes[0] = Binary::intToUint8($this->bodyType);

        Binary::$littleEndian::putUint32($makeBytes, 1, $this->bodySize);

        return $makeBytes;
    }

    public function writeTo (Writer $w): int {
        $res = $w->write($this->encodeSize()) + IO::copy($w, $this->getBodyReader());
        return $res;
    }

    public function checkParams () {
        if ($this->bodySize > 10 * 1024 * 1024) {
            throw new RuntimeException("body package is too large");
        }
    }

    /**
     * @return int
     */
    public function getBodyType (): int {
        return $this->bodyType;
    }

    public function checkLimitReadOver (): bool {
        if ($this->bodyReader instanceof LimitReader) {
            return $this->bodyReader->n <= 0;
        }

        return true;
    }

    public function logData (): array {
        $ret = [
            'type'        => $this->bodyType,
            'type string' => BodyType::Map[$this->bodyType],
            'size'        => $this->bodySize,
        ];

        if ($this->bodyType == BodyType::JSON) {
            $ret['body'] = $this->contentLog();
        }

        return $ret;
    }
}
