<?php


namespace Mainto\RpcServer\RpcServer;


use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Support\Str;
use Mainto\RpcServer\Command\Sidecar\Sidecar;
use Mainto\RpcServer\RpcServer\Definition\Controller;
use Mainto\RpcServer\RpcServer\Definition\NotDefinitionStruct;
use Mainto\RpcServer\RpcServer\Definition\RpcObject\JsonMapper;
use Mainto\RpcServer\RpcServer\Definition\RpcObject\ObjectRef;
use Mainto\RpcServer\RpcServer\Definition\Struct;
use Mainto\RpcServer\Util\ArrayHelper;
use Mainto\RpcServer\Util\ObjectMapper\MapperInterface;
use Mainto\RpcServer\Util\Types\Map;
use RuntimeException;

class RpcDefinition implements Arrayable, \Serializable {
    use ArrayHelper;

    public const EmptyType = 'EmptyParams';

    private static ?RpcDefinition $instance = null;
    /**
     * 框架版本，CI composer 发布时修改此参数
     */
    public string $frameworkVersion = 'v2.1.0';
    /**
     * @var Map<Controller>|Controller[]
     */
    public Map $controllers;
    /**
     * @var Map<Struct>|Struct[]
     */
    public Map $structs;
    /**
     * @var string
     */
    public string $serviceName;

    /**
     * @var string
     */
    public string $serviceNameSnake;

    /**
     * @var string
     */
    public string $version;

    /**
     * @var string
     */
    public string $alias = "";

    private MapperInterface $mapper;

    private function __construct () {
        $this->init();
    }

    private function init (): void {
        $this->structs = new Map();
        $this->controllers = new Map();

        $empty = new Struct();
        $empty->name = self::EmptyType;
        $empty->namespace = self::EmptyType;

        $this->structs[self::EmptyType] = $empty;
        $this->mapper = new JsonMapper();
        $this->version = get_git_branch_version(env('BASE_DIR'));
        $this->alias = Str::studly(config('rpc-server.service_alias'));
        $this->serviceName = Str::studly(config('rpc-server.service_name'));
        $this->serviceNameSnake = Str::snake(config('rpc-server.service_name'), '-');
    }

    public static function getInstance (): RpcDefinition {
        if (self::$instance === null) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    public static function newInstance (): RpcDefinition {
        return new self();
    }

    public static function clear () {
        self::$instance = null;
    }

    public function inStructCache (string $type): bool {
        return isset($this->structs[$type]);
    }

    public function getStruct (string $type): Struct {
        $struct = $this->structs->get($type);
        if (!$struct) {
            throw new RuntimeException("the struct $type not definition in rpc");
        }

        return $struct;
    }

    public function objectStruct (string $type): Struct {
        $struct = $this->structs->get($type);
        if (!$struct) {
            $struct = new Struct();
            $this->structs->set($type, $struct);

            $objectRef = ObjectRef::getRef($type);
            $struct->name = $objectRef->name;
            $struct->namespace = $objectRef->namespace;
            $struct->properties = $objectRef->properties;
        }

        return $struct;
    }

    public function instanceStruct (object $object): Struct {
        $type = get_class($object);
        if (!isset($this->structs[$type])) {
            $this->structs[$type] = $struct = new Struct();

            $objectRef = ObjectRef::getRef($object);
            $struct->name = $objectRef->name;
            $struct->namespace = $objectRef->namespace;
            $struct->properties = $objectRef->properties;
        }

        return $this->structs[$type];
    }

    public function notDefinitionStruct (string $type): NotDefinitionStruct {
        if (!isset($this->structs[$type])) {
            $this->structs[$type] = $struct = new NotDefinitionStruct();

            $shortName = class_basename($type);
            $struct->name = Str::studly($shortName).Str::studly(str_replace('.', ' ', $struct->originName));
            $struct->namespace = substr($type, 0, -strlen($shortName) - 1);
        }

        return $this->structs[$type];
    }

    /**
     * 获取所有版本信息
     * @return array
     */
    public function allVersion (): array {
        return [
            'Laravel'        => app()->version(),
            'RpcFramework'   => $this->frameworkVersion,
            'SidecarVersion' => app(Sidecar::class)->getSidecarConfig(),
            'ServiceVersion' => get_service_version(env('BASE_DIR')),
        ];
    }

    public function parse (string $path) {
        $this->mapper->map(json_decode(file_get_contents($path), true), $this);
    }

    public function getMapper (): MapperInterface {
        return $this->mapper;
    }

    public function serialize () {
        return json_encode($this);
    }

    public function unserialize ($serialized) {
        $this->init();

        (new \Mainto\RpcServer\Util\ObjectMapper\JsonMapper())->map($serialized, $this);
    }
}