<?php

namespace Mainto\RpcServer\RpcServer\Middleware\Kernel;

use Closure;
use Mainto\RpcServer\Protocol\Common\BaseType;
use Mainto\RpcServer\Protocol\Request\Request;
use Mainto\RpcServer\Tracing\Context;
use Mainto\RpcServer\Tracing\Span;
use Throwable;
use Mainto\RpcServer\Protocol\Response\Response;
use Mainto\RpcServer\RpcServer\Middleware\KernelMiddleware;
use Mainto\RpcServer\RpcServer\RpcStreamContext;
use Mainto\RpcServer\Tracing\Tracer;

class TracingKernelMiddleware implements KernelMiddleware {

    /**
     * @param RpcStreamContext $context
     * @param Closure $next
     * @return Response
     * @throws Throwable
     */
    public function handle (RpcStreamContext $context, Closure $next): Response {
        // mainType
        $mainType = BaseType::getMainType($context->request->getType());
        // 判断traceId是否为空 （7 & 8）
        if ($mainType == 7 || $mainType == 8) {
            return $next($context);
        }
        // 获取所有header
        $headers = $context->request->getAllHeaders();
        // 处理链路上下文
        $tracer = Context::extract($context->request->getTraceId(), $headers[Context::TRACE_PARENT_HEADER] ?? '');
        // 捕捉错误
        try {
            $span = $this->startSpanWithRequest($context->request);
            $response = $next($context);
            $span->end();
        } catch (Throwable $e) {
            $tracer->error($e);
            throw $e;
        } finally {
            Tracer::getInstance()->flush();
        }
        return $response;
    }

    /**
     * @param Request $request
     * @return Span
     */
    private function startSpanWithRequest (Request $request): Span {
        $span = Span::start(sprintf(
            '%s %s::%s',
            BaseType::Map[$request->getType()] ?? 'unknown',
            $request->getCallClassName(),
            $request->getCallMethodName(),
        ));
        if (!empty($request->getAllParams())) {
            $span->setAttribute('allParams', json_encode($request->getAllParams()));
        }
        if (!empty($request->getBody())) {
            $span->setAttribute('body', json_encode($request->getBody()->toArray()));
        }
        return $span;
    }
}