<?php

namespace Mainto\RpcServer\Tracing;

use Mainto\RpcServer\Tracing\Support\Zipkin\Exporter;
use Mainto\RpcServer\Tracing\Traits\Sampled;
use Throwable;

class Tracer
{
    use Sampled;

    /**
     * @var Tracer|null
     */
    private static ?Tracer $_instance = null;

    /**
     * @var IdGenerator
     */
    protected IdGenerator $idGenerator;

    /**
     * @var Span[]
     */
    protected array $spans = [];

    /**
     * @var Span|null
     */
    protected ?Span $currentSpan = null;

    /**
     * @var Source
     */
    protected Source $source;

    /**
     * @var Exporter
     */
    protected Exporter $exporter;

    /**
     * @var string
     */
    protected string $traceId = '';

    /**
     * @var string
     */
    protected string $parentSpanId = '';

    /**
     * 私有化构造方法
     */
    private function __construct()
    {
        $this->idGenerator = new IdGenerator();
        $this->source = new Source(new SdkVersion('rpc-tracing', '0.0.1'));
        $this->exporter = new Exporter();
        $this->sampled = true;
    }

    /**
     * 获取单例对象
     * @return Tracer
     */
    public static function getInstance(): Tracer
    {
        if (is_null(static::$_instance)) {
            static::$_instance = new static();
        }
        return self::$_instance;
    }

    /**
     * @return Source
     */
    public function getSource(): Source
    {
        return $this->source;
    }

    /**
     * 存储traceId
     *
     * @param string $traceId
     * @param string $parentSpanId
     * @return $this
     */
    public function setTraceId(string $traceId, string $parentSpanId = ''): self
    {
        $this->traceId = $traceId;
        $this->parentSpanId = $parentSpanId;
        return $this;
    }

    /**
     * @return string
     */
    public function getTraceId(): string
    {
        if (empty($this->traceId)) {
            $this->traceId = $this->idGenerator->generateTraceId();
        }
        return $this->traceId;
    }

    /**
     * 获取id生成器
     *
     * @return IdGenerator
     */
    public function getIdGenerator(): IdGenerator
    {
        return $this->idGenerator;
    }

    /**
     * @return string
     */
    public function getCurrentSpanId(): string
    {
        if (!empty($this->currentSpan)) {
            return $this->currentSpan->getSpanId();
        }
        return $this->parentSpanId;
    }

    /**
     * @param Span $span
     * @return $this
     */
    public function appendSpan(Span $span): self
    {
        array_push($this->spans, $span);
        $this->currentSpan = $span;
        return $this;
    }

    /**
     * @param Throwable $e
     */
    public function error(Throwable $e): void
    {
        if (empty($this->currentSpan)) {
            return;
        }

        $this->currentSpan
            ->setAttribute('error', 'true')
            ->setAttribute('error.msg', $e->getMessage())
            ->end();
    }

    /**
     * 进行推送
     */
    public function flush(): void
    {
        $this->exporter->export($this->spans, $this->source);
        $this->clear();
    }

    /**
     * 清理
     */
    private function clear()
    {
        $this->spans = [];
        $this->traceId = '';
        $this->currentSpan = null;
        $this->parentSpanId = '';
    }
}
