<?php
/**
 * Created by PhpStorm.
 * User: jiuli
 * Date: 2018/4/4
 * Time: 下午1:29
 */

namespace Mainto\RpcServer\RpcClient;


use Illuminate\Support\Facades\Log;
use Mainto\RpcServer\Exceptions\RpcInvokeException;
use Mainto\RpcServer\Exceptions\RpcRuntimeException;
use Mainto\RpcServer\Protocol\Common\Body;
use Mainto\RpcServer\Protocol\Common\Types\RequestHeaderType;
use Mainto\RpcServer\Protocol\Common\Types\ResponseHeaderType;
use Mainto\RpcServer\Protocol\Request\Extend\RequestExtendHeader;
use Mainto\RpcServer\Protocol\Request\Request;
use Mainto\RpcServer\Protocol\Response\Extend\ResponseExtendError;
use Mainto\RpcServer\RpcServer\RpcInvoke;
use Mainto\RpcServer\Tracing\Context;
use RuntimeException;
use Throwable;

class RpcClass {
    protected static array $instances = [];

    /**
     * 调用服务名
     * @var string
     */
    protected string $serviceName;
    /**
     * 调用类名
     * @var string
     */
    protected string $className;

    /**
     * @var RpcClientWeakRef
     */
    private RpcClientWeakRef $clientRef;

    /**
     * RpcClass constructor.
     * @param string $serviceName
     * @param string $className
     */
    protected function __construct (string $serviceName, string $className) {
        $this->clientRef = new RpcClientWeakRef([RpcClientFactory::RpcClassClient, $serviceName]);

        $this->serviceName = $serviceName;
        $this->className = $className;
    }

    /**
     * @return RpcClient
     */
    protected function getClient(): RpcClient {
        return $this->clientRef->getClient();
    }

    public function clearClient() {
        $this->clientRef->clearClient();
    }

    /**
     * 获得一个远程调用实例（如果有设置 Mock，获得的是 Mock 实例）
     *
     * @param $serviceName
     * @param $className
     * @return RpcClass
     */
    public static function getClass ($serviceName, $className): RpcClass {
        if (!isset(self::$instances[$serviceName][$className])) {
            if (!config('rpc-server.rpc_call')['enable']) {
                throw new RpcRuntimeException("rpc call is disabled: {$serviceName}::{$className}");
            }
            self::$instances[$serviceName][$className] = new RpcClass($serviceName, $className);
        }

        return self::$instances[$serviceName][$className];
    }

    /**
     * 获取远程调用客户端
     *
     * @param $serviceName
     */
    public static function removeRemoteClient ($serviceName): void {
        RpcClientFactory::clearRpcClassClient($serviceName);

        unset(self::$instances[$serviceName]);
    }

    /**
     * 注册一个 Mock 的类
     *
     * @param $serviceName
     * @param $className
     * @return RpcMockClass
     */
    public static function registerMockClass ($serviceName, $className): RpcMockClass {
        if (!isset(self::$instances[$serviceName][$className]) || !(self::$instances[$serviceName][$className] instanceof RpcMockClass)) {
            self::$instances[$serviceName][$className] = new RpcMockClass($serviceName, $className);
        }

        return self::$instances[$serviceName][$className];
    }

    /**
     * 反注册一个 Mock 的类
     *
     * @param $serviceName
     * @param $className
     * @return bool
     */
    public static function unregisterMockClass ($serviceName, $className): bool {
        if (isset(self::$instances[$serviceName][$className]) && self::$instances[$serviceName][$className] instanceof RpcMockClass) {
            unset(self::$instances[$serviceName][$className]);
            return true;
        }

        return false;
    }

    /**
     * 反注册所有的 Mock 类
     *
     * @return bool
     */
    public static function unregisterAllMockClass (): bool {
        foreach (self::$instances as $serviceName => $classes) {
            foreach ($classes as $className => $insClass) {
                if ($insClass instanceof RpcMockClass) {
                    unset(self::$instances[$serviceName][$className]);
                }
            }
        }

        return true;
    }

    /**
     * 函数调用
     * @param $name
     * @param $arguments
     * @return mixed
     * @throws Throwable
     */
    public function __call ($name, $arguments) {
        if (count($arguments) == 1) {
            $arguments = $arguments[0] ?? $arguments;
        }

        $request = new Request();
        $request->setCallClassAndMethod($this->className, $name);
        $request->setType(RequestHeaderType::InvokeNormalType);
        if ($arguments) {
            $request->setBody(Body::newJsonBody(json_encode($arguments)));
        }
        $request->setTraceId(RpcInvoke::getCurrentTraceId());

        $ext = new RequestExtendHeader();
        $ext->addHeader('X-Request-From', config('rpc-server.service_name'));
        $ext->addHeader(Context::TRACE_PARENT_HEADER, Context::getHeaderValue());

        if ($sessionId = RpcInvoke::getCurrentSessionId()) {
            $ext->addHeader('X-Stream-Id', $sessionId);
        }
        $request->setExtend($ext);

        // todo retry
        try {
            $response = $this->getClient()->Do($request);
            switch ($response->getType()) {
                case ResponseHeaderType::ReturnOKType:
                    if ($response->hasBody()) {
                        return $response->getBody()->getContent();
                    } else {
                        return null;
                    }
                case ResponseHeaderType::ReturnSystemErrType:
                case ResponseHeaderType::ReturnErrType:
                    $extend = $response->getExt();
                    if ($extend instanceof ResponseExtendError) {
                        throw new RpcInvokeException($extend->getErrMsg(), $extend->getErrorCode());
                    }

                    throw new RuntimeException("unknown response ext");
                default:
                    throw new RuntimeException("unknown response");
            }
        } catch (Throwable $exception) {
            if ($exception instanceof RpcRuntimeException) {
                throw $exception;
            }
            $this->clearClient();
            Log::error($exception);
            throw new RuntimeException("can not connect to server {$this->serviceName}", 0, $exception);
        }
    }
}
