<?php

namespace Mainto\RpcServer\Tracing\Support\Zipkin;

use Mainto\RpcServer\Tracing\Interfaces\SpanKind;
use Mainto\RpcServer\Tracing\Source;
use Mainto\RpcServer\Tracing\Span;

class Converter
{
    const STATUS_CODE_TAG_KEY = 'op.status_code';
    const STATUS_DESCRIPTION_TAG_KEY = 'op.status_description';

    /**
     * @param $value
     * @return string
     */
    private function sanitiseTagValue($value): string
    {
        // Casting false to string makes an empty string
        if (is_bool($value)) {
            return $value ? 'true' : 'false';
        }

        // Zipkin tags must be strings, but opentelemetry
        // accepts strings, booleans, numbers, and lists of each.
        if (is_array($value)) {
            return join(',', array_map([$this, 'sanitiseTagValue'], $value));
        }

        // Floats will lose precision if their string representation
        // is >=14 or >=17 digits, depending on PHP settings.
        // Can also throw E_RECOVERABLE_ERROR if $value is an object
        // without a __toString() method.
        // This is possible because OpenTelemetry\Trace\Span does not verify
        // setAttribute() $value input.
        return (string) $value;
    }

    /**
     * @param array $spans
     * @param Source $source
     * @return array
     */
    public function converts(array $spans, Source $source): array
    {
        // 默认值
        $convertedSpans = [];
        // 遍历循环spans
        foreach ($spans as $span) {
            // 转换
            array_push($convertedSpans, $this->convert($span, $source));
        }
        //
        return $convertedSpans;
    }

    /**
     * @param Span $span
     * @param Source $source
     * @return array
     */
    private function convert(Span $span, Source $source): array
    {
        $row = [
            'id' => $span->getSpanId(),
            'traceId' => bin2hex($span->getTraceId()),
            'parentId' => $span->getParentId(),
            'localEndpoint' => [
                'serviceName' => $source->getServiceName(),
            ],
            'kind' => SpanKind::TYPES[$span->getKind()] ?? 'SERVER',
            'name' => $span->getName(),
            'timestamp' => $span->getStart(),
            'duration' => $span->getDuration(),
            'tags' => [
                self::STATUS_CODE_TAG_KEY => $span->getStatusCode(),
                self::STATUS_DESCRIPTION_TAG_KEY => $span->getStatusMessage(),
            ],
        ];
        // 合并数组
        $row['tags'] = array_merge($row['tags'], $source->getAttributes(), $span->getAttributes());
        // 判断是否为空
        if (!empty($span->getEvents())) {
            // 初始化
            $row['annotations'] = [];
            // 遍历循环
            foreach ($span->getEvents() as $event) {
                // 追加数据
                array_push($row['annotations'], [
                    'name' => $event->getName(),
                    'timestamp' => (int) ($event->getTimestamp() / 1e6),
                ]);
            }
        }

        return $row;
    }
}