<?php

namespace Mainto\RpcServer\Tracing;

use Mainto\RpcServer\Tracing\Support\Zipkin\Exporter;
use Throwable;

class Tracer
{
    /**
     * @var Tracer|null
     */
    private static ?Tracer $_instance;

    /**
     * @var IdGenerator
     */
    protected IdGenerator $idGenerator;

    /**
     * @var string
     */
    protected string $parentSpanId;

    /**
     * @var Span[]
     */
    protected array $spans = [];

    /**
     * @var Span
     */
    protected Span $currentSpan;

    /**
     * @var Source
     */
    protected Source $source;

    /**
     * @var Exporter
     */
    protected Exporter $exporter;

    /**
     * @var string
     */
    protected string $traceId;

    /**
     * 私有化构造方法
     */
    private function __construct()
    {
        $this->idGenerator = new IdGenerator();
        $this->source = new Source(new SdkVersion('rpc-tracing', '0.0.1'));
        $this->exporter = new Exporter();
    }

    /**
     * 获取单例对象
     * @return Tracer
     */
    public static function getInstance(): Tracer
    {
        if (is_null(self::$_instance)) {
            self::$_instance = new static();
        }
        return self::$_instance;
    }

    /**
     * @return Source
     */
    public function getSource(): Source
    {
        return $this->source;
    }

    /**
     * 存储traceId
     *
     * @param string $traceId
     * @return $this
     */
    public function setTraceId(string $traceId): self
    {
        $this->traceId = $traceId;
        return $this;
    }

    /**
     * @return string
     */
    public function getTraceId(): string
    {
        if (empty($this->traceId)) {
            $this->traceId = $this->idGenerator->generateTraceId();
        }
        return $this->traceId;
    }

    /**
     * 获取id生成器
     *
     * @return IdGenerator
     */
    public function getIdGenerator(): IdGenerator
    {
        return $this->idGenerator;
    }

    /**
     * 设置当前span
     *
     * @param string $spanId
     * @return $this
     */
    public function setParentSpanId(string $spanId): self
    {
        $this->parentSpanId = $spanId;
        return $this;
    }

    /**
     * @return string
     */
    public function getParentSpanId(): string
    {
        return $this->parentSpanId;
    }

    /**
     * @param Span $span
     * @return $this
     */
    public function appendSpan(Span $span): self
    {
        array_push($this->spans, $span);
        return $this->setParentSpanId($span->getSpanId());
    }

    /**
     * @param Throwable $e
     */
    public function error(Throwable $e): void
    {
        end($this->spans)
            ->setAttribute('error', true)
            ->setAttribute('error.msg', $e->getMessage());
    }

    /**
     * 进行推送
     */
    public function flush(): void
    {
        $this->exporter->export($this->spans, $this->source);
        $this->spans = [];
        $this->traceId = '';
    }
}
