<?php

namespace Mainto\RpcServer\Tracing;

use Mainto\RpcServer\Tracing\Support\Aliyun;
use Mainto\RpcServer\Tracing\Support\Zipkin;
use Mainto\RpcServer\Tracing\Traits\Sampled;
use Throwable;

class Tracer {
    use Sampled;

    /**
     * @var array|string[]
     */
    protected static array $exporterMap = [
        'zipkin' => Zipkin\Exporter::class,
        'aliyun' => Aliyun\Exporter::class,
    ];

    /**
     * @var string
     */
    protected string $traceId = '';

    /**
     * @var Tracer|null
     */
    private static ?Tracer $_instance = null;

    /**
     * @var IdGenerator
     */
    protected IdGenerator $idGenerator;

    /**
     * @var Source
     */
    protected Source $source;

    /**
     * @var Processor
     */
    protected Processor $processor;

    /**
     * 私有化构造方法
     */
    private function __construct () {
        $this->idGenerator = new IdGenerator();
        $this->source = new Source(new SdkVersion('php-tracing', '0.0.1'));
        $this->processor = new Processor(new Aliyun\Exporter(), $this->source);
        $this->sampled = true;
    }

    /**
     * 获取单例对象
     * @return Tracer
     */
    public static function getInstance (): Tracer {
        if (is_null(static::$_instance)) {
            static::$_instance = new static();
        }
        return self::$_instance;
    }

    /**
     * @return Source
     */
    public function getSource (): Source {
        return $this->source;
    }

    /**
     * 存储traceId
     *
     * @param string $traceId
     * @return $this
     */
    public function setTraceId (string $traceId): self {
        $this->traceId = $traceId;
        return $this;
    }

    /**
     * @return string
     */
    public function getTraceId (): string {
        if (empty($this->traceId)) {
            $this->traceId = $this->idGenerator->generateTraceId();
        }
        return $this->traceId;
    }

    /**
     * 获取id生成器
     *
     * @return IdGenerator
     */
    public function getIdGenerator (): IdGenerator {
        return $this->idGenerator;
    }

    /**
     * @return Processor
     */
    public function getProcessor (): Processor {
        return $this->processor;
    }

    /**
     * @param Throwable $e
     */
    public function error (Throwable $e): void {
        // 获取当前Span
        $span = Context::currentSpan();
        if (empty($span)) {
            return;
        }

        $span->setAttribute('error', 'true')
            ->setAttribute('error.msg', $e->getMessage())
            ->end();
    }

    /**
     * 进行推送
     */
    public function flush (): void {
        $this->processor->forceFlush();
        $this->clear();
    }

    /**
     * 清理
     */
    private function clear () {
        $this->traceId = '';
        Context::setSpanContext(null);
    }
}
