<?php

namespace Mainto\RpcServer\RpcUtil\Tool\LogDriver;

use Illuminate\Support\Facades\Log;
use Mainto\RpcServer\Exceptions\RpcRuntimeException;
use Mainto\RpcServer\Protocol\Common\Body;
use Mainto\RpcServer\Protocol\Common\Types\RequestHeaderType;
use Mainto\RpcServer\Protocol\Common\Types\ResponseHeaderType;
use Mainto\RpcServer\Protocol\Request\Extend\RequestExtendHeader;
use Mainto\RpcServer\Protocol\Request\Request;
use Mainto\RpcServer\Protocol\Response\Extend\ResponseExtendError;
use Mainto\RpcServer\RpcClient\RpcClient;
use Mainto\RpcServer\RpcClient\RpcClientFactory;
use Mainto\RpcServer\RpcClient\RpcClientWeakRef;
use Mainto\RpcServer\RpcServer\RpcInvoke;
use Mainto\RpcServer\Tracing\SpanArray;
use RuntimeException;
use Throwable;

class SidecarSLS extends LogDriveAbstract {
    private static ?Request $cacheRequest = null;

    /**
     * @var RpcClientWeakRef
     */
    private RpcClientWeakRef $clientRef;

    public function __construct () {
        parent::__construct();

        if ($this->localMode) {
            throw new RuntimeException('agent sls log driver can not support local mode');
        }

        if (self::$cacheRequest === null) {
            $requestExtendHeader = new RequestExtendHeader();
            self::$cacheRequest = new Request();
            self::$cacheRequest->setType(RequestHeaderType::SidecarSLSType);
            self::$cacheRequest->setExtend($requestExtendHeader);
        }

        $this->clientRef = new RpcClientWeakRef([RpcClientFactory::SlsClient]);
    }

    /**
     * @return RpcClient
     */
    protected function getClient(): RpcClient {
        return $this->clientRef->getClient();
    }

    public function clearClient() {
        $this->clientRef->clearClient();
    }

    public function logThrowable ($request, Throwable $e, string $source = "") {
        if ($source == "") {
            $traceInfo = $this->getTraceInfo(-1);
            $source = $traceInfo['source'];
        }

        $traceStr = rpc_exception_trace($e);

        $this->send($source, "error", [
            "request"       => $this->transObject($request),
            "error_name"    => class_basename($e),
            "error_code"    => $e->getCode(),
            "error_content" => $e->getMessage(),
            "error_file"    => $e->getFile(),
            "error_line"    => $e->getLine(),
            "error_trace"   => $traceStr,
        ]);
    }

    /**
     * log throwable error log
     *
     * @param $request
     * @param $source
     * @param Throwable $e
     */
    public function logRequestThrowable (Request $request, Throwable $e, string $source = "") {
        if ($source == "") {
            $traceInfo = $this->getTraceInfo(-1);
            $source = $traceInfo['source'];
        }

        $traceStr = rpc_exception_trace($e);

        $this->send($source, "error", [
            "header"        => $request->getAllHeaders(),
            "request"       => $request->getAllParams(),
            "error_name"    => class_basename($e),
            "error_code"    => $e->getCode(),
            "error_content" => $e->getMessage(),
            "error_file"    => $e->getFile(),
            "error_line"    => $e->getLine(),
            "error_trace"   => $traceStr,
        ]);
    }

    /**
     * send message to sidecar sls service
     *
     * @param $source
     * @param $logStore
     * @param $contents
     */
    private function send (string $source, string $logStore, array $contents) {
        try {
            /** @var RequestExtendHeader $ext */
            $ext = self::$cacheRequest->getExt();
            $ext->setHeader("channel", [config('rpc-server.sidecar.sls.channel')[$logStore]]);

            $contents['method'] = $source;
            $contents['service_name'] = config('rpc-server.service_name');
            self::$cacheRequest->setBody(Body::newJsonBody(json_encode($contents)));

            self::$cacheRequest->setTraceId(RpcInvoke::getCurrentTraceId());
            $response = $this->getClient()->Do(self::$cacheRequest);
            switch ($response->getType()) {
                case ResponseHeaderType::ReturnOKType:
                    break;
                case ResponseHeaderType::ReturnSystemErrType:
                case ResponseHeaderType::ReturnErrType:
                    $extend = $response->getExt();
                    if ($extend instanceof ResponseExtendError) {
                        throw new RpcRuntimeException($extend->getErrMsg(), $extend->getErrorCode());
                    }

                    throw new RuntimeException("unknown response ext");
                default:
                    throw new RuntimeException("unknown response");
            }
        } catch (Throwable $e) {
            if ($e instanceof RpcRuntimeException) {
                throw $e;
            }
            $this->clearClient();
            Log::error($e, $contents);
            throw new RuntimeException("can not find this sls server", 0, $e);
        }
    }

    /**
     * log info message
     *
     * @param $title
     * @param $message
     */
    public function info (string $title, $message) {
        $this->log("INFO", $title, $message);
    }

    /**
     * log some message
     *
     * @param string $level
     * @param string $title
     * @param $message
     */
    protected function log (string $level, string $title, $message) {
        $traceInfo = $this->getTraceInfo();
        $this->send($traceInfo['source'], "log", [
            "offset"  => $traceInfo['file'],
            "level"   => $level,
            "title"   => $title,
            "content" => $this->transObject($message),
        ]);
    }

    /**
     * log debug message
     *
     * @param $title
     * @param $message
     */
    public function debug (string $title, $message) {
        $this->log("DEBU", $title, $message);
    }

    /**
     * log notice message
     *
     * @param $title
     * @param $message
     */
    public function notice (string $title, $message) {
        $this->log("NOTI", $title, $message);
    }

    /**
     * log warning message
     *
     * @param $title
     * @param $message
     */
    public function warning (string $title, $message) {
        $this->log("WARN", $title, $message);
    }

    /**
     * log error message
     *
     * @param $source
     * @param $title
     * @param $message
     */
    public function error (string $title, $message) {
        $this->log("ERRO", $title, $message);
    }

    /**
     * log critical message
     *
     * @param $title
     * @param $message
     */
    public function critical (string $title, $message) {
        $this->log("CRIT", $title, $message);
    }

    /**
     * @param SpanArray $spans
     */
    public function tracer(SpanArray $spans) {
        $this->send('TRACER', 'tracer', $spans->toArray());
    }
}
