<?php

namespace Mainto\RpcServer\Tracing;

use Mainto\RpcServer\Tracing\Interfaces\SpanKind;
use Mainto\RpcServer\Tracing\Traits\Attributes;
use Mainto\RpcServer\Tracing\Traits\Links;
use Mainto\RpcServer\Tracing\Traits\Events;
use Mainto\RpcServer\Tracing\Traits\Resource;
use Mainto\RpcServer\Tracing\Traits\Sampled;
use Mainto\RpcServer\Tracing\Traits\Status;
use Mainto\RpcServer\Tracing\Traits\Timer;
use Mainto\RpcServer\Util\Types\Map;

class Span {
    use Attributes, Links, Events, Resource, Timer, Status, Sampled;

    /**
     * @var string
     */
    protected string $name;

    /**
     * @var int
     */
    protected int $kind;

    /**
     * @var SpanContext
     */
    protected SpanContext $context;

    /**
     * @var Span|null
     */
    protected ?Span $parent = null;

    /**
     * @var Tracer
     */
    protected Tracer $tracer;

    /**
     * @param string $name
     * @param SpanContext|null $context
     * @param int $kind
     */
    private function __construct (string $name = '', SpanContext $context = null, int $kind = SpanKind::KIND_INTERNAL) {
        $this->tracer = Tracer::getInstance();

        $this->start = microtime(true) * $this->timeUnit;
        $this->name = $name;
        $this->kind = $kind;
        $this->statusOk();
        $this->attributes = new Map();
        $this->resource = new Map();
        // 获取当前spanContext中的span
        $this->parent = $context->getSpan();

        $this->context = new SpanContext(
            $this->tracer->getTraceId(),
            $this->tracer->getIdGenerator()->generateSpanId(),
            $context->getSpanId(),
            $this,
        );

        $this->tracer->getProcessor()->onStart($this);
        $this->setSampled($this->tracer->isSampled());
    }

    /**
     * 快速调用
     *
     * @param string $name
     * @param SpanContext|null $context
     * @param int $kind
     *
     * @return Span
     */
    public static function start (string $name, SpanContext $context = null, int $kind = SpanKind::KIND_INTERNAL): Span {
        if (is_null($context)) {
            $context = Context::getSpanContext();
        }
        if (Tracer::getInstance()->isSampled() === false) {
            return NoopSpan::getInstance($name, $context, $kind);
        }
        return new Span($name, $context, $kind);
    }

    /**
     * @return string
     */
    public function getName (): string {
        return $this->name;
    }

    /**
     * @param string $name
     * @return $this
     */
    public function setName (string $name): self {
        $this->name = $name;
        return $this;
    }

    /**
     * @return int
     */
    public function getKind (): int {
        return $this->kind;
    }

    /**
     * @return string
     */
    public function getSpanId (): string {
        return $this->context->getSpanId();
    }

    /**
     * @return string
     */
    public function getTraceId (): string {
        return $this->context->getTraceId();
    }

    /**
     * @return string
     */
    public function getParentId (): string {
        return $this->context->getParentSpanId();
    }

    /**
     * @return SpanContext
     */
    public function getContext (): SpanContext {
        return $this->context;
    }

    /**
     * @return Span|null
     */
    public function parent (): ?Span {
        return $this->parent;
    }

    /**
     * span结束
     */
    public function end (): void {
        if (empty($time) && !empty($this->end)) {
            return;
        }

        $this->end = microtime(true) * $this->timeUnit;
        $this->duration = $this->end - $this->start;

        $this->tracer->getProcessor()->onEnd($this);
    }
}
