<?php

namespace Mainto\RpcServer\RpcServer\Middleware\Kernel;

use Closure;
use Mainto\RpcServer\Protocol\Common\BaseType;
use Mainto\RpcServer\Protocol\Common\Types\RequestHeaderType;
use Mainto\RpcServer\Protocol\Request\Request;
use Mainto\RpcServer\Tracing\Context;
use Mainto\RpcServer\Tracing\Interfaces\SpanKind;
use Mainto\RpcServer\Tracing\Span;
use Throwable;
use Mainto\RpcServer\Protocol\Response\Response;
use Mainto\RpcServer\RpcServer\Middleware\KernelMiddleware;
use Mainto\RpcServer\RpcServer\RpcStreamContext;
use Mainto\RpcServer\Tracing\Tracer;

class TracingKernelMiddleware implements KernelMiddleware {

    /**
     * SpanKind 映射关系
     * @var array
     */
    private array $spanKind = [
        RequestHeaderType::UrlInvokeType      => SpanKind::KIND_SERVER,
        RequestHeaderType::InternalInvokeType => SpanKind::KIND_INTERNAL,
        RequestHeaderType::MessageType        => SpanKind::KIND_CONSUMER,
        RequestHeaderType::CronType           => SpanKind::KIND_CONSUMER,
        RequestHeaderType::WebsocketType      => SpanKind::KIND_SERVER,
    ];

    /**
     * @param RpcStreamContext $context
     * @param Closure $next
     * @return Response
     * @throws Throwable
     */
    public function handle (RpcStreamContext $context, Closure $next): Response {
        // 解码处理类型
        $mainType = BaseType::getMainType($context->request->getType());
        // 判断是否为不需要进行链路处理的消息
        if ($mainType == RequestHeaderType::SidecarType || $mainType == RequestHeaderType::CheckType) {
            return $next($context);
        }
        // 获取所有header
        $headers = $context->request->getAllHeaders();
        // 处理链路上下文
        $tracer = Context::extract($context->request->getTraceId(), $headers[Context::TRACE_PARENT_HEADER] ?? '');
        // 捕捉错误
        try {
            $span = $this->startSpanWithContext($context, $mainType);
            $response = $next($context);
            $span->end();
        } catch (Throwable $e) {
            $tracer->error($e);
            throw $e;
        } finally {
            $tracer->flush();
        }
        return $response;
    }

    /**
     * @param RpcStreamContext $context
     * @param int $mainType
     * @return Span
     */
    private function startSpanWithContext (RpcStreamContext $context, int $mainType): Span {
        // span名称
        $spanName = sprintf(
            '%s %s::%s',
            BaseType::Map[$context->request->getType()] ?? 'Unknown',
            $context->request->getCallClassName(),
            $context->request->getCallMethodName(),
        );
        // 开启span
        $span = Span::start($spanName, null, $this->spanKind[$mainType] ?? SpanKind::KIND_SERVER);
        $span->setAttribute('pid', $context->pid);
        // 获取所有参数，不为空则记录
        $allParams = $context->request->getAllParams();
        if (!empty($allParams)) {
            $span->setAttribute('request.params', json_encode($allParams));
        }
        return $span;
    }
}