<?php

namespace Mainto\RpcServer\Tracing\Extend\GuzzleHttp;

use Mainto\RpcServer\RpcUtil\JsonHelper;
use Mainto\RpcServer\Tracing\Context;
use Mainto\RpcServer\Tracing\Interfaces\SpanKind;
use Mainto\RpcServer\Tracing\Span;
use Psr\Http\Message\RequestInterface;
use Psr\Http\Message\ResponseInterface;

/**
 * 拦截器对象
 */
class Middleware {
    /**
     * 返回请求链路拦截
     * @return callable
     */
    public static function traceMiddleware (): callable {
        return static function (callable $handler): callable {
            $ctx = Context::getSpanContext();
            return static function (RequestInterface $request, array $options) use ($handler, $ctx) {
                $spanName = sprintf(
                    '%s %s%s',
                    $request->getMethod(),
                    $request->getUri()->getHost(),
                    $request->getUri()->getPath(),
                );
                $span = Span::start($spanName, $ctx, SpanKind::KIND_CLIENT);
                $span->setAttribute('request.query', $request->getUri()->getQuery());
                $span->setAttribute('request.port', (string)$request->getUri()->getPort());
                $span->setAttribute('request.scheme', $request->getUri()->getScheme());
                $span->setAttribute('request.headers', JsonHelper::encodeWithUnescapedAndError($request->getHeaders()));
                if ($content = $request->getBody()->getContents()) {
                    $span->setAttribute('request.body', $content);
                }
                // 增加链路header
                $request = $request->withHeader(Context::TRACE_PARENT_HEADER, Context::getHeaderValue());
                try {
                    if (empty($options['http_errors'])) {
                        return $handler($request, $options);
                    }
                    return $handler($request, $options)->then(
                        static function (ResponseInterface $response) use ($request, $span) {
                            $span->setAttribute('response.code', $response->getStatusCode());
                            $span->setAttribute(
                                'response.headers',
                                JsonHelper::encodeWithUnescapedAndError($response->getHeaders())
                            );
                            if ($content = $response->getBody()->getContents()) {
                                $span->setAttribute('response.body', $content);
                            }
                            $response->getBody()->rewind();
                            return $response;
                        }
                    );
                } finally {
                    $span->end();
                }
            };
        };
    }
}