<?php

namespace Mainto\RpcServer\Providers;

use GuzzleHttp\HandlerStack;
use Illuminate\Cache\Events\CacheHit;
use Illuminate\Cache\Events\CacheMissed;
use Illuminate\Cache\Events\KeyWritten;
use Illuminate\Database\Events\QueryExecuted;
use Illuminate\Foundation\Support\Providers\EventServiceProvider;
use Illuminate\Support\Facades\Event;
use Mainto\RpcServer\RpcUtil\JsonHelper;
use Mainto\RpcServer\Tracing\Context;
use Mainto\RpcServer\Tracing\Interfaces\SpanKind;
use Mainto\RpcServer\Tracing\Listeners\CacheHitListener;
use Mainto\RpcServer\Tracing\Listeners\CacheMissedListener;
use Mainto\RpcServer\Tracing\Listeners\DatabaseQueryListener;
use Mainto\RpcServer\Tracing\Listeners\KeyWrittenListener;
use GuzzleHttp\Client;
use Mainto\RpcServer\Tracing\Span;
use Psr\Http\Message\RequestInterface;
use Psr\Http\Message\ResponseInterface;

class TracingProvider extends EventServiceProvider {
    /**
     * The event listener mappings for the application.
     *
     * @var array
     */
    protected $listen = [
        QueryExecuted::class => DatabaseQueryListener::class,
        CacheHit::class      => CacheHitListener::class,
        CacheMissed::class   => CacheMissedListener::class,
        KeyWritten::class    => KeyWrittenListener::class,
    ];

    /**
     * Register any events for your application.
     *
     * @return void
     */
    public function boot () {
        foreach ($this->listen as $event => $listener) {
            Event::listen($event, $listener);
        }
    }

    /**
     *
     */
    public function register () {
        $this->app->bind(Client::class, function ($app, array $config = []) {
            if (!isset($config['handler'])) {
                $config['handler'] = HandlerStack::create();
            }
            $config['handler']->push($this->requestMiddleware(), 'trace');
            return new Client($config);
        });
    }

    /**
     * @return callable
     */
    private function requestMiddleware (): callable {
        return static function (callable $handler): callable {
            $ctx = Context::getSpanContext();
            return static function (RequestInterface $request, array $options) use ($handler, $ctx) {
                $spanName = sprintf(
                    '%s %s%s',
                    $request->getMethod(),
                    $request->getUri()->getHost(),
                    $request->getUri()->getPath(),
                );
                $span = Span::start($spanName, $ctx, SpanKind::KIND_CLIENT);
                $span->setAttribute('request.query', $request->getUri()->getQuery());
                $span->setAttribute('request.port', (string)$request->getUri()->getPort());
                $span->setAttribute('request.scheme', $request->getUri()->getScheme());
                $span->setAttribute('request.headers', JsonHelper::encodeWithUnescapedAndError($request->getHeaders()));
                if ($content = $request->getBody()->getContents()) {
                    $span->setAttribute('request.body', $content);
                }
                // 增加链路header
                $request = $request->withHeader(Context::TRACE_PARENT_HEADER, Context::getHeaderValue());
                try {
                    if (empty($options['http_errors'])) {
                        return $handler($request, $options);
                    }
                    return $handler($request, $options)->then(
                        static function (ResponseInterface $response) use ($request, $span) {
                            $span->setAttribute('response.code', $response->getStatusCode());
                            $span->setAttribute(
                                'response.headers',
                                JsonHelper::encodeWithUnescapedAndError($response->getHeaders())
                            );
                            if ($content = $response->getBody()->getContents()) {
                                $span->setAttribute('response.body', $content);
                            }
                            $response->getBody()->rewind();
                            return $response;
                        }
                    );
                } finally {
                    $span->end();
                }
            };
        };
    }
}
