<?php


namespace Mainto\RpcServer\RpcServer;


use Closure;
use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Contracts\Support\Jsonable;
use Mainto\RpcServer\Exceptions\RpcNotSupportException;
use Mainto\RpcServer\Exceptions\RpcRuntimeException;
use Mainto\RpcServer\Protocol\Request\Request;
use Mainto\RpcServer\Protocol\Response\Extend\ResponseExtendHeader;
use Mainto\RpcServer\Protocol\Response\Response;
use Mainto\RpcServer\RpcServer\Middleware\MethodMiddleware;
use ReflectionException;

class RpcInvoke {

    private static ?RpcContext $currentContext = null;

    /**
     * @var array ['methodName' => ['class' => 'MethodMiddlewareInstance', 'args' => []]]
     */
    protected static array $methodMiddlewaresMap = [];

    /**
     * @var array|null ['methodName' => 'callback']
     */
    protected static ?array $methodInvokerMap = null;

    public static function resetMethodMiddleware (string $fullMethodName) {
        unset(self::$methodMiddlewaresMap[$fullMethodName]);
        unset(self::$methodInvokerMap[$fullMethodName]);
    }

    public static function addMethodMiddleware (string $fullMethodName, MethodMiddleware $middleware, $args = []) {
        self::$methodMiddlewaresMap[$fullMethodName][] = [
            'class' => $middleware,
            'args'  => $args,
        ];
    }

    public static function newRpcContext(Request $request): RpcContext {
        return self::$currentContext = new RpcContext($request);
    }

    public static function init() {
        self::initMethodInvoker();
    }

    /**
     * @param RpcContext $context
     * @return mixed
     */
    public static function invokeThroughMiddleware (RpcContext $context) {
        $method = $context->getRequest()->getCallClassName().'::'.$context->getRequest()->getCallMethodName();

        if (isset(self::$methodInvokerMap[$method])) {
            return (self::$methodInvokerMap[$method])($context);
        }

        return self::invokeBase($context);
    }

    protected static function initMethodInvoker () {
        self::$methodInvokerMap = [];
        foreach (self::$methodMiddlewaresMap as $fullMethodName => $middlewares) {
            self::$methodInvokerMap[$fullMethodName] = array_reduce(
                array_reverse($middlewares),
                fn(Closure $next, $middle) => fn(RpcContext $context) => $middle['class']->handle($context, $next, $middle['args']),
                function (RpcContext $context) {
                    return self::invokeBase($context);
                },
            );
        }
    }

    /**
     * @param RpcContext $context
     * @return mixed
     */
    public static function invokeBase (RpcContext $context) {
        $method = $context->getMethod();
        $context->setDeclareNames($method->getRequestDeclareNames());

        try {
            return $method->getRefInstance()->invokeArgs($context->getController()->getInstance(), $context->getParameters());
        } catch (ReflectionException $exception) {
            throw new \RuntimeException($exception->getMessage());
        }
    }

    /**
     * @return RpcContext|null
     */
    public static function getCurrentContext (): ?RpcContext {
        return self::$currentContext;
    }

    /**
     * @return RpcContext|null
     */
    public static function getCurrentSessionId (): ?string {
        if (self::$currentContext) {
            return self::$currentContext->getSessionId();
        }

        return null;
    }

    /**
     * @return RpcContext|null
     */
    public static function getCurrentTraceId (): ?string {
        if (self::$currentContext) {
            return self::$currentContext->getTraceId();
        }

        return "";
    }

    /**
     * @param Request $request
     * @return Response
     */
    public static function invoke (Request $request): Response {
        $context = self::newRpcContext($request);

        $ret = RpcInvoke::invokeThroughMiddleware($context);

        if ($ret instanceof Response) {
            return $ret;
        }

        RpcResponseHelper::$cacheResponse->resetBodyAndExt();

        // body
        $body = null;
        switch ($context->getMethod()->resSerialize) {
            case 'Json':
            case 'Jsonp':
                $body = RpcResponseHelper::$cacheJsonBody;
                if ($ret instanceof Jsonable) {
                    $encode = (string)$ret->toJson();
                } else {
                    $encode = json_encode($ret instanceof Arrayable ? $ret->toArray() : $ret);

                    if ($encode === false) {
                        throw new RpcRuntimeException("the response can not convert to json: ".json_last_error_msg());
                    }
                }

                $body->setBodyBytes($encode);
                break;
            case 'Raw':
                $body = RpcResponseHelper::$cacheRawBody;
                $body->setBodyBytes((string)$ret);
                break;
            default:
                throw new RpcNotSupportException("body type not support: ".$context->getMethod()->resSerialize);
        }

        $response = RpcResponseHelper::$cacheResponse->useReturnOKType();
        $response->setBody($body);

        // responseHeaders
        $responseHeaders = $context->getResponseHeader();
        if ($responseHeaders) {
            $ext = new ResponseExtendHeader();
            foreach ($responseHeaders as $key => $value) {
                $ext->setHeader($key, (array)$value);
            }

            $response->setExtend($ext);
        }

        return $response;
    }
}