<?php
/**
 * Created by PhpStorm.
 * User: PHPStorm
 * Date: 18-10-22
 * Time: 下午3:17
 */

namespace Mainto\RpcServer\Providers;

use Doctrine\Common\Annotations\AnnotationReader;
use Doctrine\Common\Annotations\AnnotationRegistry;
use Mainto\RpcServer\Base\Controller;
use Mainto\RpcServer\Command\RpcChildCommand;
use Mainto\RpcServer\Command\RpcDefinitionCommand;
use Mainto\RpcServer\Command\RpcServerCommand;
use Mainto\RpcServer\Command\RpcSidecarCommand;
use Mainto\RpcServer\RpcClient\RpcClass;
use Mainto\RpcServer\RpcServer\RpcInvoke;
use Mainto\RpcServer\RpcServer\RpcResponseHelper;
use Mainto\RpcServer\RpcServer\RpcRouter;
use Mainto\RpcServer\RpcServer\RpcSessionHandlerInterface;
use Mainto\RpcServer\RpcUtil\RpcMessage;
use Mainto\RpcServer\RpcUtil\RpcMessageQueue;
use Mainto\RpcServer\RpcUtil\Tool\RpcLog;
use Mainto\RpcServer\RpcUtil\Tool\RpcMutexProviderInterface;
use Mainto\RpcServer\Util\Language;
use Mainto\RpcServer\Util\ObjectMapper\JsonMapper;

class ServiceProvider extends \Illuminate\Support\ServiceProvider {
    /**
     * Bootstrap the application services.
     *
     * @return void
     */
    public function boot () {
        if ($this->app->runningInConsole()) {

            $this->publishes([
                __DIR__.'/../config.php' => config_path('rpc-server.php'),
            ], 'config');
        }
    }

    /**
     * Register the application services.
     *
     * @return void
     */
    public function register () {
        $this->mergeConfigFrom(__DIR__.'/../config.php', 'rpc-server');

        config()->set('rpc-server.environment', rpc_environment(gethostname()));
        config()->set('rpc-server.gateway_url', rpc_gateway_url(config('rpc-server.environment')));

        $this->app->bind('command.rpc.sidecar', RpcSidecarCommand::class);
        $this->app->bind('command.rpc.server', RpcServerCommand::class);
        $this->app->bind('command.rpc.child', RpcChildCommand::class);
        $this->app->bind('command.rpc.definition', RpcDefinitionCommand::class);

        $this->commands([
            'command.rpc.sidecar',
            'command.rpc.server',
            'command.rpc.child',
            'command.rpc.definition',
        ]);

        $this->app->singleton('rpc-log', function () {
            return RpcLog::getInstance();
        });

        $this->app->singleton('rpc-server', fn() => RpcRouter::getInstance());

        $loader = new RpcAnnotationLoader();
        // todo remove
        AnnotationRegistry::registerLoader([$loader, "loadClass"]);

        $this->app->singleton(AnnotationReader::class, function () {
            return new AnnotationReader();
        });

        if (config('rpc-server.mutex_driver') == 'gear-lock') {
            $this->app->singleton(RpcMutexProviderInterface::class, function () {
                return new class implements RpcMutexProviderInterface {
                    private static function getLockCli () {
                        return RpcClass::getClass("CoreGear", "Lock");
                    }

                    public static function getLock (string $lockName, int $expireTime = 600) {
                        return self::getLockCli()->getLock([
                            "lockName"   => $lockName,
                            "expireTime" => $expireTime,
                        ]);
                    }

                    public static function unLock (string $lockName, string $password): bool {
                        return self::getLockCli()->unlock([
                            "lockName" => $lockName,
                            "password" => $password,
                        ]);
                    }
                };
            });
        } else {
            (new RpcMutexDefaultProvider($this->app))->register();
        }

        if (config('rpc-server.session_driver') == 'gear-store') {
            $this->app->singleton(RpcSessionHandlerInterface::class, function () {
                return new class implements RpcSessionHandlerInterface {
                    private function getStore () {
                        return RpcClass::getClass("CoreGear", "Store");
                    }

                    public function getSessionStringBySessionId ($sessionId): ?string {
                        return $this->getStore()->get([
                            "key" => $sessionId,
                        ]);
                    }

                    public function updateSession ($sessionId, $sessionString, $seconds): void {
                        $this->getStore()->set([
                            "key"        => $sessionId,
                            "value"      => $sessionString,
                            "expireTime" => $seconds,
                        ]);
                    }

                    public function deleteSession ($sessionId): void {
                        $this->getStore()->delete([
                            "key" => $sessionId,
                        ]);
                    }
                };
            });
        } else {
            (new SessionDefaultProvider($this->app))->register();
        }

        // init service
        Language::init();
        RpcResponseHelper::init();

        app('events')->listen('bootstrapped: Illuminate\Foundation\Bootstrap\BootProviders', fn() => RpcInvoke::init());

        $this->app->singleton(RpcMessageQueue::class, fn() => RpcMessageQueue::getInstance());
        $this->app->singleton(RpcMessage::class, fn() => RpcMessage::getInstance());
        $this->app->singleton('json_mapper', fn() => new JsonMapper());

        $this->autoBind();
    }

    public function autoBind() {
        $appDir = app_path();
        if (RpcRouter::getAppDir()) {
            $appDir = RpcRouter::getAppDir();
        }

        foreach (get_classes($appDir) as $class) {
            if (is_subclass_of($class, Controller::class)) {
                $classImplements = class_implements($class);
                foreach ($classImplements as $implement) {
                    $this->app->singleton($implement, $class);
                }
            }
        }
    }
}
