<?php


namespace Mainto\RpcServer\Util\ObjectMapper;


use Illuminate\Support\Str;
use Mainto\RpcServer\Util\Language;
use Mainto\RpcServer\Util\Types\Map;
use ReflectionClass;
use ReflectionException;
use ReflectionMethod;
use ReflectionProperty;

class ClassRef implements PropertiesInterface {
    public function getProperties (): array {
        return $this->properties;
    }
    /**
     * @var PropertyRef[]
     */
    public array $properties;

    /**
     * @var string[]
     */
    public array $useClasses;
    public string $namespace;

    public ReflectionClass $class;
    public string $name;

    /**
     * ClassRef constructor.
     * @param $object
     * @throws ReflectionException
     */
    public function __construct ($object) {
        $this->class = new ReflectionClass($object);

        $this->useClasses = self::parseUseClasses($this->class->getFileName());
        $this->namespace = $this->class->getNamespaceName();
        $this->name = $this->class->getShortName();
        $this->loadProperties();
    }

    public function loadProperties() {
        $methods = array_flip(array_map(fn($method) => $method->name, $this->class->getMethods(ReflectionMethod::IS_PUBLIC)));

        $this->properties = array_map(function (ReflectionProperty $reflectionProperty) use ($methods) {
            $setMethod = 'set'.Str::studly($reflectionProperty->name);

            $property = new PropertyRef();
            [$property->type, $_] = $this->getType($reflectionProperty, $this);
            $property->name = $reflectionProperty->name;
            $property->setterMethod = isset($methods[$setMethod]) ? $setMethod : '';

            return $property;
        }, $this->class->getProperties(ReflectionProperty::IS_PUBLIC));
    }

    public static function parseUseClasses ($classFile): array {
        $useClasses = [];

        $fp = fopen($classFile, 'r');
        while (($line = ltrim(fgets($fp))) !== false) {
            if (starts_with($line, 'class')) {
                break;
            }
            if (starts_with($line, 'use')) {
                $useStr = rtrim(ltrim($line, 'use '), "; \n");
                if (str_contains($useStr, 'as')) {
                    [$fullClass, $useName] = explode('as', $useStr);
                    $useClasses[trim($useName)] = trim($fullClass);
                } else {
                    $useClasses[class_basename($useStr)] = $useStr;
                }
            }
        }
        fclose($fp);

        return $useClasses;
    }

    protected function getType (ReflectionProperty $property, self $classRef) {
        if ($property->getType()) {
            $type = $property->getType()->getName();
            switch ($type) {
                case 'array':
                    return ArrayTypeParser::parseArrayProperty($property->getDocComment(), $classRef->useClasses, $classRef->namespace);
                case Map::class:
                    return MapTypeParser::parseMapProperty($property->getDocComment(), $classRef->useClasses, $classRef->namespace);
                default:
                    return [$type, $type];
            }
        }

        return ['mixed', null];
    }


}